import React from 'react';
import dynamic from "next/dynamic";
const Chart = dynamic(() => import("react-apexcharts"), { ssr: false });
import { useTheme } from '@mui/material/styles';
import { MenuItem, Grid, Stack, Typography, Button, Avatar, Box } from '@mui/material';
import { IconGridDots } from '@tabler/icons-react';
import DashboardCard from '../../shared/DashboardCard';
import CustomSelect from '../../forms/theme-elements/CustomSelect';
import SkeletonRevenueUpdatesTwoCard from '../skeleton/RevenueUpdatesTwoCard';
import { ApexOptions } from 'apexcharts';


interface RevenueupdatestwoCardProps {
  isLoading?: boolean;
}

const RevenueUpdates = ({ isLoading }: RevenueupdatestwoCardProps) => {
  const [month, setMonth] = React.useState('1');

  const handleChange = (event: React.ChangeEvent<HTMLInputElement>) => {
    setMonth(event.target.value);
  };

  // chart color
  const theme = useTheme();
  const primary = theme.palette.primary.main;
  const secondary = theme.palette.secondary.main;

  // chart
  const optionscolumnchart: ApexOptions = {
    chart: {
      type: 'bar',
      fontFamily: "'Plus Jakarta Sans', sans-serif;",
      foreColor: '#adb0bb',
      toolbar: {
        show: true,
      },
      height: 360,
      stacked: true,
    },
    colors: [primary, secondary],
    plotOptions: {
      bar: {
        horizontal: false,
        barHeight: '60%',
        columnWidth: '20%',
        borderRadius: 6,
        borderRadiusApplication: 'end',
        borderRadiusWhenStacked: 'all',
      },
    },

    stroke: {
      show: false,
    },
    dataLabels: {
      enabled: false,
    },
    legend: {
      show: false,
    },
    grid: {
      borderColor: 'rgba(0,0,0,0.1)',
      strokeDashArray: 3,
      xaxis: {
        lines: {
          show: false,
        },
      },
    },
    yaxis: {
      min: -5,
      max: 5,
      tickAmount: 4,
    },
    xaxis: {
      categories: ['16/08', '17/08', '18/08', '19/08', '20/08', '21/08', '22/08'],
      axisBorder: {
        show: false,
      },
    },
    tooltip: {
      theme: theme.palette.mode === 'dark' ? 'dark' : 'light',
      fillSeriesColor: false,
    },
  };
  const seriescolumnchart = [
    {
      name: 'Eanings this month',
      data: [1.5, 2.7, 2.2, 3.6, 1.5, 1.0],
    },
    {
      name: 'Expense this month',
      data: [-1.8, -1.1, -2.5, -1.5, -0.6, -1.8],
    },
  ];

  return (
    <>
      {
        isLoading ? (
          <SkeletonRevenueUpdatesTwoCard />
        ) : (
          <DashboardCard
            title="Revenue Updates"
            subtitle="Overview of Profit"
            action={
              <CustomSelect
                labelId="month-dd"
                id="month-dd"
                size="small"
                value={month}
                onChange={handleChange}
              >
                <MenuItem value={1}>March 2025</MenuItem>
                <MenuItem value={2}>April 2025</MenuItem>
                <MenuItem value={3}>May 2025</MenuItem>
              </CustomSelect>
            }
          >
            <Grid container spacing={3}>
              {/* column */}
              <Grid
                size={{
                  xs: 12,
                  sm: 8
                }}>
                <Box className="rounded-bars">
                  <Chart
                    options={optionscolumnchart}
                    series={seriescolumnchart}
                    type="bar"
                    height={360}
                    width={"100%"}
                  />
                </Box>
              </Grid>
              {/* column */}
              <Grid
                size={{
                  xs: 12,
                  sm: 4
                }}>
                <Stack spacing={3} mt={3}>
                  <Stack direction="row" spacing={2} alignItems="center">
                    <Box
                      width={40}
                      height={40}
                      bgcolor="primary.light"
                      display="flex"
                      alignItems="center"
                      justifyContent="center"
                    >
                      <Typography color="primary" variant="h6" display="flex">
                        <IconGridDots size={24} />
                      </Typography>
                    </Box>
                    <Box>
                      <Typography variant="h3" fontWeight="700">
                        $63,489.50
                      </Typography>
                      <Typography variant="subtitle2" color="textSecondary">
                        Total Earnings
                      </Typography>
                    </Box>
                  </Stack>
                </Stack>
                <Stack spacing={3} my={5}>
                  <Stack direction="row" spacing={2} sx={{ display: "flex", alignItems: "center" }}>
                    <Avatar
                      sx={{ width: 9, mt: 1, height: 9, bgcolor: primary, svg: { display: 'none' } }}
                    ></Avatar>
                    <Box >
                      <Typography variant="subtitle1" color="textSecondary">
                        Earnings this month
                      </Typography>
                      <Typography variant="h5">$48,820</Typography>
                    </Box>
                  </Stack>
                  <Stack direction="row" spacing={2} sx={{ display: "flex", alignItems: "center" }}>
                    <Avatar
                      sx={{ width: 9, mt: 1, height: 9, bgcolor: secondary, svg: { display: 'none' } }}
                    ></Avatar>
                    <Box>
                      <Typography variant="subtitle1" color="textSecondary">
                        Expense this month
                      </Typography>
                      <Typography variant="h5">$26,498</Typography>
                    </Box>
                  </Stack>
                </Stack>
                <Button color="primary" variant="contained" fullWidth>
                  View Full Report
                </Button>
              </Grid>
            </Grid>
          </DashboardCard>
        )}
    </>
  );
};

export default RevenueUpdates;
