import { useEffect, useMemo, useState } from "react";
import { useTable, usePagination, useGlobalFilter, useSortBy } from "react-table";
import { Button, Table } from "react-bootstrap";
import axios from "axios";

export const BasicTable = () => {
    const [data, setData] = useState([]);
    const [totalCount, setTotalCount] = useState(0);
    const [pageSize, setPageSize] = useState(10);  // Default page size
    const [pageIndex, setPageIndex] = useState<any>(0);  // Explicit page index state
    const [isLoading, setIsLoading] = useState(false);
    const [error, setError] = useState('');

    const fetchTxns = async (page = 1, limit = 10) => {
        try {
            setIsLoading(true);
            const token = localStorage.getItem('token') ?? '';
            const res = await axios.post(`/api/rcllcClicks`, { page, limit, token });
            const json = await res.data;
            setData(json.data.rows);
            setTotalCount(json.data.count);
        } catch (err) {
            console.error('Error fetching transactions:', err);
            setError("Failed to fetch data.");
        } finally {
            setIsLoading(false);
        }
    };

    // Define your columns
    const columns = useMemo(() => [
        {
            Header: "Sl No.",
            accessor: "id",
            Cell: ({ row }: any) => row.index + 1,
        },
        {
            Header: "Time",
            accessor: "created_at",
            Cell: ({ row }: any) => row.original.created_at,
        },
        {
            Header: "IP",
            accessor: "ip",
            Cell: ({ row }: any) => row.original.ip,
        },
        {
            Header: "Button",
            accessor: "button",
            Cell: ({ row }: any) => row.original.button,
        },
    ], []);

    // Table instance
    const tableInstance = useTable(
        {
            columns,
            data,
            pageCount: Math.ceil(totalCount / pageSize),
            manualPagination: true,  // We are manually controlling pagination
            initialState: { pageIndex },  // Use pageIndex as initial state
        },
        useGlobalFilter,
        useSortBy,
        usePagination
    );

    const {
        getTableProps,
        getTableBodyProps,
        headerGroups,
        rows,
        prepareRow,
        state,

        canNextPage,
        canPreviousPage,
        nextPage,
        previousPage,
        setPageSize: tableSetPageSize,
        gotoPage,
    }: any = tableInstance;

    const { pageIndex: tablePageIndex } = state;

    // Fetch data when pageIndex or pageSize changes
    useEffect(() => {
        fetchTxns(tablePageIndex + 1, pageSize);  // tablePageIndex + 1 because API expects 1-based index
    }, [tablePageIndex, pageSize]);  // Effect only runs when pageIndex or pageSize changes

    useEffect(() => {
        setPageIndex(tablePageIndex);  // Ensure the pageIndex in the state is synced with the table instance
    }, [tablePageIndex]);

    const handlePageChange = (newPageIndex: number) => {
        setPageIndex(newPageIndex);  // Update page index in local state
        gotoPage(newPageIndex);  // Use the `gotoPage` method to navigate to a specific page
    };

    return (
        <>
            {/* Top Bar */}
            <div className="d-flex mb-3">
                <div className="form-group">
                    <label className="me-2">Show</label>
                    <select
                        className="selectpage border me-1"
                        value={pageSize}
                        onChange={(e) => {
                            const newPageSize = Number(e.target.value);
                            setPageSize(newPageSize);  // Update page size
                            tableSetPageSize(newPageSize);  // Update the table's internal state
                        }}
                    >
                        {[10, 25, 50].map((size) => (
                            <option key={size} value={size}>
                                {size}
                            </option>
                        ))}
                    </select>{" "}
                    entries
                </div>
            </div>

            {/* Table */}
            <Table bordered hover size="sm" {...getTableProps()}>
                <thead>
                    {headerGroups.map((headerGroup: any) => (
                        <tr {...headerGroup.getHeaderGroupProps()} key={headerGroup.id}>
                            {headerGroup.headers.map((column: any) => (
                                <th {...column.getHeaderProps(column.getSortByToggleProps())} key={column.id} className="px-3 py-2">
                                    {column.render("Header")}
                                    <span>{column.isSorted ? (column.isSortedDesc ? " 🔽" : " 🔼") : ""}</span>
                                </th>
                            ))}
                        </tr>
                    ))}
                </thead>
                <tbody {...getTableBodyProps()}>
                    {rows.length === 0 ? (
                        <tr>
                            <td colSpan={columns.length} className="text-center py-4">
                                No data available
                            </td>
                        </tr>
                    ) : (
                        rows.map((row: any) => {
                            prepareRow(row);
                            return (
                                <tr {...row.getRowProps()} key={row.id}>
                                    {row.cells.map((cell: any) => (
                                        <td {...cell.getCellProps()} key={cell.column.id} className="px-3 py-2">
                                            {cell.render("Cell")}
                                        </td>
                                    ))}
                                </tr>
                            );
                        })
                    )}
                </tbody>
            </Table>

            {/* Pagination */}
            <div className="d-block d-sm-flex mt-4">
                <span className="ms-sm-auto">
                    <Button
                        variant=""
                        className="btn-outline-light tablebutton me-2 d-sm-inline d-block my-1"
                        onClick={() => handlePageChange(0)}  // Go to the first page
                        disabled={!canPreviousPage}
                    >
                        {"<<"}
                    </Button>
                    <Button
                        variant=""
                        className="btn-outline-light tablebutton me-2 my-1"
                        onClick={() => handlePageChange(tablePageIndex - 1)}  // Go to the previous page
                        disabled={!canPreviousPage}
                    >
                        {"<"}
                    </Button>
                    <Button
                        variant=""
                        className="btn-outline-light tablebutton me-2 my-1"
                        onClick={() => handlePageChange(tablePageIndex + 1)}  // Go to the next page
                        disabled={!canNextPage}
                    >
                        {">"}
                    </Button>
                    <Button
                        variant=""
                        className="btn-outline-light tablebutton me-2 my-1"
                        onClick={() => { handlePageChange(Math.ceil(totalCount / pageSize) - 1) }}  // Go to the last page
                        disabled={!canNextPage}
                    >
                        {">>"}
                    </Button>
                </span>
            </div>
        </>
    );
};
