const { Admin, Logins, LinkedAccounts, Transactions, Schedule, Days, LeaveApplication } = require('../models/Associations')
const { Op, where, fn, col, literal } = require('sequelize');
const bcrypt = require('bcryptjs');
const jwt = require('jsonwebtoken');
const axios = require('axios');
const path = require('path')
const date = require('date-and-time');
const moment = require('moment-timezone');
const crypto = require('crypto');
const qs = require('qs');
const fs = require('fs');
const useragent = require("useragent");
const FormData = require('form-data');
const db = require('../config/db.config')
const mysql = require('mysql2');
const puppeteer = require("puppeteer");
require('dotenv').config()
const html_to_pdf = require("html-pdf-node");
const dayjs = require("dayjs");
const PayoutMethods = require('../models/PayoutMethods');

// async function isWithinScheduleWindow(adminId) {
//     const now = moment().tz("Asia/Kolkata");

//     const startOfDay = now.clone().startOf("day").format("YYYY-MM-DD HH:mm:ss");
//     const endOfDay = now.clone().endOf("day").format("YYYY-MM-DD HH:mm:ss");

//     const schedules = await Schedule.findAll({
//         where: {
//             adminid: adminId,
//             start_time: {
//                 [Op.between]: [startOfDay, endOfDay],
//             },
//         },
//     });

//     // ✅ IMPORTANT: If no schedule exists → ALLOW login
//     if (!schedules.length) {
//         return { allowed: true };
//     }

//     for (const s of schedules) {
//         const start = moment
//             .tz(s.start_time, "Asia/Kolkata")
//             .subtract(3, "hours");

//         const end = moment
//             .tz(s.end_time, "Asia/Kolkata")
//             .add(3, "hours");

//         if (now.isBetween(start, end, undefined, "[]")) {
//             return { allowed: true };
//         }
//     }

//     return {
//         allowed: false,
//         reason:
//             "Login allowed only within 3 hours before start and 3 hours after end of scheduled shift",
//     };
// }
async function isWithinScheduleWindow(adminId) {
    const now = moment().tz("Asia/Kolkata");

    const startOfDay = now.clone().startOf("day").format("YYYY-MM-DD HH:mm:ss");
    const endOfDay = now.clone().endOf("day").format("YYYY-MM-DD HH:mm:ss");

    const schedules = await Schedule.findAll({
        where: {
            adminid: adminId,
            start_time: {
                [Op.between]: [startOfDay, endOfDay],
            },
        },
    });

    // ✅ No schedule → allow login, no lateness
    if (!schedules.length) {
        return { allowed: true, lateByMinutes: 0 };
    }

    for (const s of schedules) {
        const start = moment.tz(s.start_time, "Asia/Kolkata");
        const end = moment.tz(s.end_time, "Asia/Kolkata");

        const windowStart = start.clone().subtract(3, "hours");
        const windowEnd = end.clone().add(3, "hours");

        if (now.isBetween(windowStart, windowEnd, undefined, "[]")) {
            const lateByMinutes = now.isAfter(start)
                ? now.diff(start, "minutes")
                : 0;

            return {
                allowed: true,
                lateByMinutes,
                scheduleStart: start,
            };
        }
    }

    return {
        allowed: false,
        reason:
            "Login allowed only within 3 hours before start and 3 hours after end of scheduled shift",
    };
}
function formatLateDuration(totalMinutes) {
    if (totalMinutes <= 0) return "";

    const hours = Math.floor(totalMinutes / 60);
    const minutes = totalMinutes % 60;

    let parts = [];

    if (hours > 0) {
        parts.push(`${hours} hour${hours > 1 ? "s" : ""}`);
    }

    if (minutes > 0) {
        parts.push(`${minutes} minute${minutes > 1 ? "s" : ""}`);
    }

    return parts.join(" ");
}
exports.login = async (req, res) => {
    try {
        const { username, password, recaptcha } = req.body;
        if (!username || !password) {
            return res.json({ status: 'error', message: 'Username / Password is required' })
        }
        const secret = '6LcpKxMsAAAAAC9Q9lr-RlukPuyMfI97TZpRgR7q';
        const googleRes = await axios.post(
            `https://www.google.com/recaptcha/api/siteverify`,
            {},
            {
                params: {
                    secret,
                    response: recaptcha,
                },
            }
        );

        if (!googleRes.data.success) {
            return res.status(400).json({ status: "error", message: "Invalid reCAPTCHA" });
        }

        const admin = await Admin.findOne({ where: { username: username } })
        if (!admin) {
            return res.json({ status: 'error', message: 'User not found' })
        }
        const compatibleHash = admin.password.replace(/^\$2y\$/, '$2b$');

        const passwordMatch = await bcrypt.compare(password, compatibleHash);

        if (!passwordMatch) {
            return res.json({ status: 'error', message: 'Incorrect password' });
        }
        const jwtData = {
            id: admin.id,
            first: admin.first,
            last: admin.last,
            email: admin.email,
            phone: admin.phone,
            role: admin.role,
            username: admin.username
        };
        const token = jwt.sign(jwtData, process.env.JWT_SECRET, { algorithm: 'HS256' });
        res.json({ status: 'success', accessToken: token, user: jwtData })
    } catch (error) {
        return res.json({ status: 'error', message: error?.message })
    }
}
exports.makeLogin = async (req, res) => {
    let user = null;

    try {
        /* =====================
           AUTH VALIDATION
        ===================== */
        const authHeader = req.headers['authorization'] || req.headers['Authorization'];
        if (!authHeader) {
            return res.json({ status: 'error', message: 'Authorization header is missing' });
        }

        const token = authHeader.split(' ')[1];
        if (!token) {
            return res.json({ status: 'error', message: 'Token is missing' });
        }

        const decoded = jwt.verify(token, process.env.JWT_SECRET);
        const userId = decoded.id;

        /* =====================
           FETCH USER
        ===================== */
        user = await Admin.findOne({ where: { id: userId } });
        if (!user) {
            return res.json({ status: 'error', message: 'User not found' });
        }

        /* =====================
           🔒 LOGIN LOCK CHECK
        ===================== */
        if (user.login_in_progress === 1) {
            return res.json({
                status: "error",
                message: "Login already in progress. Please wait.",
            });
        }

        /* =====================
           🔐 ACQUIRE LOCK
        ===================== */
        await user.update({ login_in_progress: 1 });

        /* =====================
           MOBILE LOGIN RESTRICTION
        ===================== */
        const { screen_width, user_agent } = req.body;

        const scheduleCheck = await isWithinScheduleWindow(user.id);
        if (!scheduleCheck.allowed) {
            return res.json({
                status: "error",
                message: scheduleCheck.reason,
            });
        }

        if (user.mobile_login !== '1') {
            const ua = (user_agent || '').toLowerCase();
            const screenWidth = screen_width || 0;

            const isDesktop =
                ua.includes("windows") ||
                ua.includes("macintosh") ||
                ua.includes("linux");

            const isMobileUA =
                ua.includes("iphone") ||
                ua.includes("android") ||
                ua.includes("mobile") ||
                ua.includes("ipad") ||
                ua.includes("opera mini") ||
                ua.includes("iemobile");

            const isMobile =
                isMobileUA ||
                !isDesktop ||
                (screenWidth && screenWidth < 768);

            if (isMobile) {
                return res.json({
                    status: "error",
                    message: "Mobile login is not permitted.",
                    ua: user_agent,
                    width: screenWidth
                });
            }
        }

        /* =====================
           CREATE LOGIN ENTRY
        ===================== */
        const login = await Logins.create({
            userid: user.id,
            type: 1,
            logtime: moment().tz("America/Denver").format("YYYY-MM-DD HH:mm:ss"),
        });

        /* =====================
           SMS MESSAGE
        ===================== */
        const mobileNumber = `918617368299,19158457387,${user.phone}`;

        const lateMinutes = scheduleCheck.lateByMinutes || 0;
        const lateText = formatLateDuration(lateMinutes);

        let message = `Employee: ${user.username} has signed in at: ${moment()
            .tz("America/Denver")
            .format("YYYY-MM-DD h:mm A")}`;

        if (lateText) {
            message += ` (Late by ${lateText})`;
        }

        const payload = {
            phone: mobileNumber,
            message: message,
        };

        const smsResponse = await axios.post(
            "https://bend.logiclane.tech/api/sms",
            payload,
            {
                headers: {
                    "Content-Type": "application/json",
                    Authorization:
                        "Bearer 4d0f394ec46be1c61d203a4df09da3277aa8c520d922533bf332c7db2c261f61",
                },
            }
        );

        if (smsResponse.data?.status === "success") {
            login.msg_response = JSON.stringify(smsResponse.data);
            await login.save();

            return res.json({
                status: "success",
                message: "Signed In",
            });
        }

        return res.json({
            status: "error",
            message: "Failed to send SMS",
            sms_response: smsResponse.data,
        });

    } catch (error) {
        console.error("Login error:", error);
        return res.status(500).json({
            status: "error",
            message: "Server error",
            details: error.message,
        });

    } finally {
        /* =====================
           🔓 ALWAYS RELEASE LOCK
        ===================== */
        if (user) {
            await user.update({ login_in_progress: 0 }).catch(() => { });
        }
    }
};
exports.makeLogout = async (req, res) => {
    let user = null;

    try {
        /* =====================
           AUTH VALIDATION
        ===================== */
        const authHeader = req.headers['authorization'] || req.headers['Authorization'];
        if (!authHeader) {
            return res.json({
                status: 'error',
                message: 'Authorization header is missing',
            });
        }

        const token = authHeader.split(' ')[1];
        if (!token) {
            return res.json({
                status: 'error',
                message: 'Token is missing',
            });
        }

        const decoded = jwt.verify(token, process.env.JWT_SECRET);
        const userId = decoded.id;

        /* =====================
           FETCH USER
        ===================== */
        user = await Admin.findOne({ where: { id: userId } });
        if (!user) {
            return res.json({
                status: 'error',
                message: 'User not found',
            });
        }

        /* =====================
           🔒 LOCK CHECK
        ===================== */
        if (user.login_in_progress === 1) {
            return res.json({
                status: "error",
                message: "Another action is already in progress. Please wait.",
            });
        }

        /* =====================
           🔐 ACQUIRE LOCK
        ===================== */
        await user.update({ login_in_progress: 1 });

        /* =====================
           SCHEDULE CHECK
        ===================== */
        const scheduleCheck = await isWithinScheduleWindow(user.id);
        if (!scheduleCheck.allowed) {
            return res.json({
                status: "error",
                message: scheduleCheck.reason,
            });
        }

        /* =====================
           MOBILE LOGIN RESTRICTION
        ===================== */
        if (user.mobile_login !== '1') {
            const { screen_width, user_agent } = req.body;

            const ua = (user_agent || '').toLowerCase();
            const screenWidth = screen_width || 0;

            const isDesktop =
                ua.includes("windows") ||
                ua.includes("macintosh") ||
                ua.includes("linux");

            const isMobileUA =
                ua.includes("iphone") ||
                ua.includes("android") ||
                ua.includes("mobile") ||
                ua.includes("ipad") ||
                ua.includes("opera mini") ||
                ua.includes("iemobile");

            const isMobile =
                isMobileUA ||
                !isDesktop ||
                (screenWidth && screenWidth < 768);

            if (isMobile) {
                return res.json({
                    status: "error",
                    message: "Mobile login is not permitted.",
                    ua: user_agent,
                    width: screenWidth
                });
            }
        }

        /* =====================
           CREATE LOGOUT ENTRY
        ===================== */
        const login = await Logins.create({
            userid: user.id,
            type: 2, // logout
            logtime: moment().tz("America/Denver").format("YYYY-MM-DD HH:mm:ss"),
        });

        /* =====================
           SMS NOTIFICATION
        ===================== */
        const mobileNumber = `918617368299,19158457387,${user.phone}`;
        const message = `Employee: ${user.username} has signed out at: ${moment()
            .tz("America/Denver")
            .format("YYYY-MM-DD h:mm A")}`;

        const payload = {
            phone: mobileNumber,
            message: message,
        };

        const smsResponse = await axios.post(
            "https://bend.logiclane.tech/api/sms",
            payload,
            {
                headers: {
                    "Content-Type": "application/json",
                    Authorization:
                        "Bearer 4d0f394ec46be1c61d203a4df09da3277aa8c520d922533bf332c7db2c261f61",
                },
            }
        );

        if (smsResponse.data?.status === "success") {
            login.msg_response = JSON.stringify(smsResponse.data);
            await login.save();

            return res.json({
                status: "success",
                message: "Signed Out",
            });
        }

        return res.json({
            status: "error",
            message: "Failed to send SMS",
            sms_response: smsResponse.data,
        });

    } catch (error) {
        console.error("Logout error:", error);
        return res.status(500).json({
            status: "error",
            message: "Server error",
            details: error.message,
        });

    } finally {
        /* =====================
           🔓 ALWAYS RELEASE LOCK
        ===================== */
        if (user) {
            await user.update({ login_in_progress: 0 }).catch(() => { });
        }
    }
};
// exports.makeLogin = async (req, res) => {
//     try {
//         const authHeader = req.headers['authorization'] || req.headers['Authorization'];
//         if (!authHeader) {

//             return res.json({
//                 status: 'error',
//                 message: 'Authorization header is missing',
//             });
//         }

//         const token = authHeader.split(' ')[1];
//         if (!token) {

//             return res.json({
//                 status: 'error',
//                 message: 'Token is missing',
//             });
//         }

//         const decoded = jwt.verify(token, process.env.JWT_SECRET);
//         const userId = decoded.id;
//         // const { userId } = req.body; // assuming JWT middleware attaches user to req.user
//         const { screen_width, user_agent } = req.body;
//         const user = await Admin.findOne({ where: { id: userId } })
//         // MOBILE LOGIN RESTRICTION
//         const scheduleCheck = await isWithinScheduleWindow(user.id);

//         if (!scheduleCheck.allowed) {
//             return res.json({
//                 status: "error",
//                 message: scheduleCheck.reason,
//             });
//         }

//         if (user.mobile_login != '1') {
//             const ua = user_agent.toLowerCase();
//             const screenWidth = screen_width;

//             // detect desktop
//             const isDesktop =
//                 ua.includes("windows") ||
//                 ua.includes("macintosh") ||
//                 ua.includes("linux");

//             // detect mobile
//             const isMobileUA =
//                 ua.includes("iphone") ||
//                 ua.includes("android") ||
//                 ua.includes("mobile") ||
//                 ua.includes("ipad") ||
//                 ua.includes("opera mini") ||
//                 ua.includes("iemobile");

//             const isMobile =
//                 isMobileUA ||
//                 !isDesktop ||
//                 (screenWidth && screenWidth < 768);

//             // reject mobile
//             if (isMobile) {
//                 return res.json({
//                     status: "error",
//                     message: "Mobile login is not permitted.",
//                     ua: user_agent,
//                     width: screenWidth
//                 });
//             }
//         }

//         // CREATE LOGIN ENTRY
//         const login = await Logins.create({
//             userid: user.id,
//             type: 1,
//             logtime: moment().tz("America/Denver").format("YYYY-MM-DD HH:mm:ss"),
//         });

//         // SMS PARAMETERS
//         const mobileNumber = `918617368299,19158457387,${user.phone}`;
//         // const message = `Employee: ${user.username} has signed in at: ${moment()
//         //     .tz("America/Denver")
//         //     .format("YYYY-MM-DD h:mm A")}`;
//         const lateMinutes = scheduleCheck.lateByMinutes || 0;
//         const lateText = formatLateDuration(lateMinutes);

//         let message = `Employee: ${user.username} has signed in at: ${moment()
//             .tz("America/Denver")
//             .format("YYYY-MM-DD h:mm A")}`;

//         if (lateText) {
//             message += ` (Late by ${lateText})`;
//         }
//         // SEND SMS (similar to cURL)
//         const payload = {
//             phone: mobileNumber,
//             message: message,
//         };

//         const smsResponse = await axios.post(
//             "https://bend.logiclane.tech/api/sms",
//             payload,
//             {
//                 headers: {
//                     "Content-Type": "application/json",
//                     Authorization:
//                         "Bearer 4d0f394ec46be1c61d203a4df09da3277aa8c520d922533bf332c7db2c261f61",
//                 },
//             }
//         );

//         if (smsResponse.data?.status === "success") {
//             login.msg_response = JSON.stringify(smsResponse.data);
//             await login.save();

//             return res.json({
//                 status: "success",
//                 message: "Signed In",
//             });
//         } else {
//             return res.json({
//                 status: "error",
//                 message: "Failed to send SMS",
//                 sms_response: smsResponse.data,
//             });
//         }
//     } catch (error) {
//         console.error("Login error:", error);
//         return res.status(500).json({
//             status: "error",
//             message: "Server error",
//             details: error.message,
//         });
//     }
// };
// exports.makeLogout = async (req, res) => {
//     try {
//         const authHeader = req.headers['authorization'] || req.headers['Authorization'];
//         if (!authHeader) {

//             return res.json({
//                 status: 'error',
//                 message: 'Authorization header is missing',
//             });
//         }

//         const token = authHeader.split(' ')[1];
//         if (!token) {

//             return res.json({
//                 status: 'error',
//                 message: 'Token is missing',
//             });
//         }

//         const decoded = jwt.verify(token, process.env.JWT_SECRET);
//         const userId = decoded.id;
//         // const { userId } = req.body; // assuming JWT middleware attaches user to req.user
//         const user = await Admin.findOne({ where: { id: userId } })
//         // MOBILE LOGIN RESTRICTION
//         const scheduleCheck = await isWithinScheduleWindow(user.id);

//         if (!scheduleCheck.allowed) {
//             return res.json({
//                 status: "error",
//                 message: scheduleCheck.reason,
//             });
//         }
//         if (user.mobile_login != '1') {
//             const { screen_width, user_agent } = req.body;

//             const ua = user_agent.toLowerCase();
//             const screenWidth = screen_width;

//             // detect desktop
//             const isDesktop =
//                 ua.includes("windows") ||
//                 ua.includes("macintosh") ||
//                 ua.includes("linux");

//             // detect mobile
//             const isMobileUA =
//                 ua.includes("iphone") ||
//                 ua.includes("android") ||
//                 ua.includes("mobile") ||
//                 ua.includes("ipad") ||
//                 ua.includes("opera mini") ||
//                 ua.includes("iemobile");

//             const isMobile =
//                 isMobileUA ||
//                 !isDesktop ||
//                 (screenWidth && screenWidth < 768);

//             // reject mobile
//             if (isMobile) {
//                 return res.json({
//                     status: "error",
//                     message: "Mobile login is not permitted.",
//                     ua: user_agent,
//                     width: screenWidth
//                 });
//             }
//         }

//         // CREATE LOGIN ENTRY
//         const login = await Logins.create({
//             userid: user.id,
//             type: 2,
//             logtime: moment().tz("America/Denver").format("YYYY-MM-DD HH:mm:ss"),
//         });

//         // SMS PARAMETERS
//         const mobileNumber = `918617368299,19158457387,${user.phone}`;
//         const message = `Employee: ${user.username} has signed out at: ${moment()
//             .tz("America/Denver")
//             .format("YYYY-MM-DD h:mm A")}`;

//         // SEND SMS (similar to cURL)
//         const payload = {
//             phone: mobileNumber,
//             message: message,
//         };

//         const smsResponse = await axios.post(
//             "https://bend.logiclane.tech/api/sms",
//             payload,
//             {
//                 headers: {
//                     "Content-Type": "application/json",
//                     Authorization:
//                         "Bearer 4d0f394ec46be1c61d203a4df09da3277aa8c520d922533bf332c7db2c261f61",
//                 },
//             }
//         );

//         if (smsResponse.data?.status === "success") {
//             login.msg_response = JSON.stringify(smsResponse.data);
//             await login.save();

//             return res.json({
//                 status: "success",
//                 message: "Signed In",
//             });
//         } else {
//             return res.json({
//                 status: "error",
//                 message: "Failed to send SMS",
//                 sms_response: smsResponse.data,
//             });
//         }
//     } catch (error) {
//         console.error("Login error:", error);
//         return res.status(500).json({
//             status: "error",
//             message: "Server error",
//             details: error.message,
//         });
//     }
// };
exports.updateTime = async (req, res) => {
    try {
        const { id, time } = req.body;

        if (!id || !time) {
            return res.json({
                status: "error",
                message: "ID and time are required",
            });
        }

        // Format time like Laravel: 'Y-m-d H:i:s'
        const formattedTime = moment(time).format("YYYY-MM-DD HH:mm:ss");

        const login = await Logins.findOne({ where: { id } });

        if (!login) {
            return res.json({
                status: "error",
                message: "Log not found",
            });
        }

        const update = await login.update({ logtime: formattedTime });

        if (update) {
            return res.json({
                status: "success",
                message: "Log updated",
            });
        } else {
            return res.json({
                status: "error",
                message: "Log update failed",
            });
        }
    } catch (error) {
        console.error("Update time error:", error);
        return res.status(500).json({
            status: "error",
            message: "Server error",
            details: error.message,
        });
    }
};
exports.deleteTime = async (req, res) => {
    try {
        const { id } = req.body;

        if (!id) {
            return res.json({
                status: "error",
                message: "ID is required",
            });
        }

        const login = await Logins.findOne({ where: { id } });

        if (!login) {
            return res.json({
                status: "error",
                message: "Log not found",
            });
        }

        const deleted = await login.destroy();

        if (deleted) {
            return res.json({
                status: "success",
                message: "Log deleted",
            });
        } else {
            return res.json({
                status: "error",
                message: "Log deletion failed",
            });
        }
    } catch (error) {
        console.error("Delete time error:", error);

        return res.status(500).json({
            status: "error",
            message: "Server error",
            details: error.message,
        });
    }
};
exports.deleteEmployee = async (req, res) => {
    try {
        const { id } = req.body;

        if (!id) {
            return res.json({
                status: "error",
                message: "Employee ID is required",
            });
        }

        const admin = await Admin.findOne({ where: { id } });

        if (!admin) {
            return res.json({
                status: "error",
                message: "Employee not found",
            });
        }

        const deleted = await admin.destroy();

        if (deleted) {
            return res.json({
                status: "success",
                message: "Employee deleted",
            });
        } else {
            return res.json({
                status: "error",
                message: "Employee deletion failed",
            });
        }
    } catch (error) {
        console.error("Delete employee error:", error);

        return res.status(500).json({
            status: "error",
            message: "Server error",
            details: error.message,
        });
    }
};

function generateRandomString(length = 10) {
    return crypto.randomBytes(length).toString("hex").slice(0, length);
}
exports.addEmployee = async (req, res) => {
    try {
        const authHeader = req.headers['authorization'] || req.headers['Authorization'];
        if (!authHeader) {

            return res.json({
                status: 'error',
                message: 'Authorization header is missing',
            });
        }

        const token = authHeader.split(' ')[1];
        if (!token) {

            return res.json({
                status: 'error',
                message: 'Token is missing',
            });
        }

        const decoded = jwt.verify(token, process.env.JWT_SECRET);
        const userId = decoded.id;
        const user = await Admin.findOne({ where: { id: userId } }); // assuming JWT middleware adds user to req.user

        // ONLY super-admin can add employee
        if (user.role !== "Super") {
            return res.status(401).json({
                status: "error",
                message: "Unauthorized",
            });
        }

        const { first, last, username, email, phone } = req.body;

        // Required field check
        if (!first || !username || !email || !phone) {
            return res.json({
                status: "error",
                message: "Missing required fields",
            });
        }

        const firstx = first;
        const lastx = last || "";
        const usernamex = username;
        const emailx = email;
        const phonex = phone;

        // Generate random password
        const passwordPlain = generateRandomString();
        const passwordHash = await bcrypt.hash(passwordPlain, 10);

        // Create employee in DB
        const newAdmin = await Admin.create({
            username: usernamex,
            password: passwordHash,
            first: firstx,
            last: lastx,
            email: emailx,
            phone: phonex,
            role: "Admin",
            pref: "magiclink",
        });

        if (!newAdmin) {
            return res.json({
                status: "error",
                message: "Failed to create employee",
            });
        }

        // SEND EMAIL VIA API
        const payload = {
            subject: "Account Credentials",
            email: email,
            name: username,
            mail: `
                <p>Hi ${username},<br>Your password for login is: ${passwordPlain}</p>
                <p>URL: EManage</p>
            `,
        };

        await axios.post("https://bend.logiclane.tech/api/email", payload, {
            headers: {
                "Content-Type": "application/json",
                Authorization:
                    "Bearer 4d0f394ec46be1c61d203a4df09da3277aa8c520d922533bf332c7db2c261f61",
            },
        });

        return res.json({
            status: "success",
            message: "Employee added successfully",
        });
    } catch (error) {
        console.error("Add Employee Error:", error);
        return res.status(500).json({
            status: "error",
            message: "Server error",
            details: error.message,
        });
    }
};
exports.employeeLogins = async (req, res) => {
    try {
        const authHeader = req.headers['authorization'] || req.headers['Authorization'];
        if (!authHeader) {

            return res.json({
                status: 'error',
                message: 'Authorization header is missing',
            });
        }

        const token = authHeader.split(' ')[1];
        if (!token) {

            return res.json({
                status: 'error',
                message: 'Token is missing',
            });
        }

        const decoded = jwt.verify(token, process.env.JWT_SECRET);
        const userId = decoded.id;
        const user = await Admin.findOne({ where: { id: userId } });

        // Only SUPER can access
        if (user.role !== "Super") {
            return res.status(401).json({
                status: "error",
                message: "Unauthorized",
            });
        }

        // Fetch all admins for dropdown list
        const admins = await Admin.findAll();

        const { from, to, admin } = req.query;

        // If filters are not provided, return admin list only
        if (!from || !to || !admin) {
            return res.json({
                status: "success",
                admins,
                data: [],
                total_time: null,
            });
        }

        // Find selected admin
        const selectedAdmin = await Admin.findOne({
            where: { username: admin },
        });

        if (!selectedAdmin) {
            return res.json({
                status: "error",
                message: "Admin not found",
            });
        }

        // Time range logic (same as Laravel)
        const from_date = moment(from)
            .subtract(15, "minutes")
            .format("YYYY-MM-DD HH:mm:ss");

        const to_date = moment(to)
            .add(15, "minutes")
            .format("YYYY-MM-DD HH:mm:ss");

        // Fetch logins with admin relation
        const adminLogins = await Logins.findAll({
            where: {
                userid: selectedAdmin.id,
                logtime: {
                    [Op.between]: [from_date, to_date],
                },
            },
            // include: [
            //     {
            //         model: Admin,
            //         as: "admin",
            //     },
            // ],
            order: [["id", "DESC"]],
        });

        const timeArr = adminLogins.map(l => l.toJSON());

        // Calculate total time (you must provide function)
        const total_time = calculateLoginTime(timeArr);

        return res.json({
            status: "success",
            // admins,
            cashier: admin,
            data: timeArr,
            total_time,
        });
    } catch (error) {
        console.error("Employee logins error:", error);
        return res.status(500).json({
            status: "error",
            message: "Server error",
            details: error.message,
        });
    }
};
exports.getAdmin = async (req, res) => {
    try {
        const authHeader = req.headers['authorization'] || req.headers['Authorization'];
        if (!authHeader) {
            return res.json({ status: 'error', message: 'Authorization header is missing' });
        }

        const token = authHeader.split(' ')[1];
        if (!token) {
            return res.json({ status: 'error', message: 'Token is missing' });
        }

        const decoded = jwt.verify(token, process.env.JWT_SECRET);
        const userId = decoded.id;

        // Fetch user + latest login
        const user = await Admin.findOne({
            where: { id: userId },
            attributes: { exclude: ['password'] },
            include: [
                {
                    model: Logins,
                    as: 'logins',
                    separate: true,
                    limit: 1,
                    order: [['created_at', 'DESC']]
                }
            ]
        });

        if (!user) {
            return res.json({ status: 'error', message: 'User not found' });
        }

        // Latest login
        const latestLogin = user.logins?.length > 0 ? user.logins[0] : null;

        // Remove logins array before returning
        delete user.dataValues.logins;

        let signedIn = 0;
        let logs = [];
        let logTime = null;

        if (latestLogin) {
            // SIGNED IN OUT LOGIC
            if (latestLogin.type === 1) signedIn = 1;   // SIGN-IN
            else if (latestLogin.type === 2) signedIn = 2; // SIGN-OUT

            logTime = latestLogin.logtime;

            // Fetch today's logs based on America/Denver
            const today = moment().tz('America/Denver').format('YYYY-MM-DD');

            logs = await Logins.findAll({
                where: {
                    userid: user.id,
                    logtime: { [Op.startsWith]: today }
                },
                order: [['id', 'DESC']]
            });
        }

        return res.json({
            status: 'success',
            admin: {
                ...user.dataValues,
                latest_login: latestLogin,
                signedIn,
                logTime,
                logs
            }
        });

    } catch (error) {
        console.error("Report Error:", error);
        return res.status(500).json({
            status: "error",
            message: "Server error",
            details: error.message,
        });
    }
};
exports.getAdmins = async (req, res) => {
    try {
        const admins = await Admin.findAll({ attributes: { exclude: ['password'] } })
        res.json({ status: 'success', admins })
    } catch (error) {
        console.error("Add Employee Error:", error);
        return res.status(500).json({
            status: "error",
            message: "Server error",
            details: error.message,
        });
    }
}
exports.getAdminsWithLogins = async (req, res) => {
    try {
        const admins = await Admin.findAll({
            include: [
                {
                    model: Logins,
                    as: 'logins',
                    separate: true,
                    limit: 10,
                    order: [['created_at', 'DESC']]
                }
            ],
            attributes: { exclude: ['password'] }
        })
        res.json({ status: 'success', admins })
    } catch (error) {
        console.error("Add Employee Error:", error);
        return res.status(500).json({
            status: "error",
            message: "Server error",
            details: error.message,
        });
    }
}
exports.getAdminsWorkDays = async (req, res) => {
    try {
        const month = req.query.month;
        const adminFilterRaw = req.query.admin ?? null;
        const adminFilter = adminFilterRaw ? Number(adminFilterRaw) : null;
        const timezone = req.query.timezone ?? 'MST'
        const reqTz =
            timezone === "MST"
                ? "America/Denver"
                : "Asia/Kolkata";
        if (!month || !/^\d{4}-\d{2}$/.test(month)) {
            return res.json({
                status: "error",
                message: "Valid month format required: YYYY-MM",
            });
        }

        // Month window in America/Denver
        // const startOfMonth = moment
        //     .tz(`${month}-01`, "YYYY-MM-DD", "America/Denver")
        //     .startOf("month");
        // const endOfMonth = startOfMonth.clone().endOf("month");
        const monthStartTz = moment
            .tz(`${month}-01`, "YYYY-MM-DD", reqTz)
            .startOf("month");

        const monthEndTz = monthStartTz.clone().endOf("month");

        const dbStartUtc = monthStartTz.clone().subtract(1, "day").utc();
        const dbEndUtc = monthEndTz.clone().add(1, "day").utc();

        // FETCH LOGS ----------------------
        // const logsWhere = {
        //     logtime: {
        //         [Op.between]: [
        //             startOfMonth.clone().subtract(1, "day").format("YYYY-MM-DD HH:mm:ss"),
        //             endOfMonth.clone().add(1, "day").format("YYYY-MM-DD HH:mm:ss"),
        //         ],
        //     },
        // };

        // if (adminFilter) logsWhere.userid = adminFilter;
        const logsWhere = {
            logtime: {
                [Op.between]: [
                    dbStartUtc.format("YYYY-MM-DD HH:mm:ss"),
                    dbEndUtc.format("YYYY-MM-DD HH:mm:ss"),
                ],
            },
        };

        if (adminFilter) logsWhere.userid = adminFilter;

        const logs = await Logins.findAll({
            where: logsWhere,
            order: [["created_at", "ASC"]],
        });

        // FETCH ADMINS ---------------------
        const admins = await Admin.findAll({
            attributes: ["id", "username", "first", "last"],
            where: adminFilter ? { id: adminFilter } : undefined,
        });

        // GROUP LOGS BY ADMIN -------------
        const logsByAdmin = {};
        logs.forEach((log) => {
            if (!logsByAdmin[log.userid]) logsByAdmin[log.userid] = [];
            logsByAdmin[log.userid].push(log);
        });

        const output = {};

        // BUILD SHIFTS ---------------------
        for (const admin of admins) {
            const adminLogs = logsByAdmin[admin.id] || [];
            if (!adminLogs.length) continue;

            let lastLogin = null;
            const periods = [];

            for (const entry of adminLogs) {
                if (entry.type === 1) {
                    lastLogin = entry.created_at;
                } else if (entry.type === 2) {
                    if (lastLogin) {
                        periods.push({ in: lastLogin, out: entry.created_at });
                        lastLogin = null;
                    }
                }
            }

            // If login exists without logout → treat as active shift
            if (lastLogin) {
                periods.push({
                    in: lastLogin, // already UTC
                    out: moment.utc().format("YYYY-MM-DD HH:mm:ss"), // 🔥 FIX
                });
            }


            // EXPAND INTO DAYS ---------------------
            for (const period of periods) {
                const periodStart = moment.tz(period.in, "America/Denver");
                const periodEnd = moment.tz(period.out, "America/Denver");

                const periodStartUtc = moment.utc(period.in);
                const periodEndUtc = moment.utc(period.out);

                // let cursor = periodStart.clone().startOf("day");
                let cursor = periodStartUtc.clone().tz(reqTz).startOf("day");
                const periodEndTz = periodEndUtc.clone().tz(reqTz);

                // while (cursor.isSameOrBefore(periodEnd, "day")) {
                //     const dayString = cursor.format("YYYY-MM-DD");

                //     if (
                //         cursor.isSameOrAfter(startOfMonth, "day") &&
                //         cursor.isSameOrBefore(endOfMonth, "day")
                //     ) {
                //         if (!output[dayString]) output[dayString] = [];

                //         // Convert returned times to Asia/Kolkata
                //         const inKolkata = moment
                //             .tz(period.in, "YYYY-MM-DD HH:mm:ss", "America/Denver")
                //             .tz("Asia/Kolkata")
                //             .format("YYYY-MM-DD HH:mm:ss");

                //         const outKolkata = moment
                //             .tz(period.out, "YYYY-MM-DD HH:mm:ss", "America/Denver")
                //             .tz("Asia/Kolkata")
                //             .format("YYYY-MM-DD HH:mm:ss");

                //         output[dayString].push({
                //             id: admin.id,
                //             name:
                //                 admin.username ||
                //                 [admin.first, admin.last].filter(Boolean).join(" ") ||
                //                 `Admin ${admin.id}`,
                //             period: {
                //                 in: inKolkata,
                //                 out: outKolkata,
                //             },
                //         });
                //     }

                //     cursor = cursor.add(1, "day");
                // }
                while (cursor.isSameOrBefore(periodEndTz, "day")) {
                    if (
                        cursor.isSameOrAfter(monthStartTz, "day") &&
                        cursor.isSameOrBefore(monthEndTz, "day")
                    ) {
                        const dayKey = cursor.format("YYYY-MM-DD");

                        if (!output[dayKey]) output[dayKey] = [];

                        output[dayKey].push({
                            id: admin.id,
                            name:
                                admin.username ||
                                [admin.first, admin.last].filter(Boolean).join(" ") ||
                                `Admin ${admin.id}`,
                            period: {
                                in: moment.utc(period.in).tz(reqTz).format("YYYY-MM-DD HH:mm:ss"),
                                out: moment.utc(period.out).tz(reqTz).format("YYYY-MM-DD HH:mm:ss"),
                            },
                        });
                    }

                    cursor.add(1, "day");
                }
            }
        }

        return res.json({ status: "success", days: output });
    } catch (error) {
        console.error("Work Days API Error:", error);
        return res.status(500).json({
            status: "error",
            message: "Server error",
            details: error.message,
        });
    }
};

exports.updateMobileLogin = async (req, res) => {
    try {
        const authHeader = req.headers['authorization'] || req.headers['Authorization'];
        if (!authHeader) {

            return res.json({
                status: 'error',
                message: 'Authorization header is missing',
            });
        }

        const token = authHeader.split(' ')[1];
        if (!token) {

            return res.json({
                status: 'error',
                message: 'Token is missing',
            });
        }

        const decoded = jwt.verify(token, process.env.JWT_SECRET);
        const userId = decoded.id;
        const user = await Admin.findOne({ where: { id: userId } });
        console.log(userId)
        // Only Super admin allowed
        if (user.role !== "Super") {
            return res.status(401).json({
                status: "error",
                message: "Unauthorized",
            });
        }

        const { username, status } = req.body;
        console.log(username, status)
        if (!username || !status) {
            return res.json({
                status: "error",
                message: "Username and status are required",
            });
        }

        const settings = await Admin.findOne({ where: { username } });

        if (!settings) {
            console.log('here')
            return res.json({
                status: "error",
                message: "User not found",
            });
        }

        const updated = await settings.update({ mobile_login: status });

        if (updated) {
            console.log('here2')

            return res.json({
                status: "success",
                login_status: status,
                username: username,
            });
        } else {
            console.log('here3')

            return res.json({
                status: "error",
                message: "Update failed",
            });
        }
    } catch (error) {
        console.log('here4')

        console.error("updateMobileLogin error:", error);

        return res.status(500).json({
            status: "error",
            message: "Server error",
            details: error.message,
        });
    }
};
function calculateLoginTime(records) {
    let totalSeconds = 0;

    // Sort strictly by time
    records.sort((a, b) =>
        a.logtime.valueOf() - b.logtime.valueOf()
    );

    let pendingLogin = null;

    for (const record of records) {
        if (!record.logtime || !record.logtime.isValid()) continue;

        if (record.type == 1) {
            // LOGIN
            if (!pendingLogin) {
                pendingLogin = record.logtime.clone();
            }
            // duplicate login → ignore
        }

        else if (record.type == 2) {
            // LOGOUT
            if (pendingLogin) {
                const diff = record.logtime.diff(pendingLogin, "seconds");
                if (diff > 0) totalSeconds += diff;
                pendingLogin = null;
            }
            // logout without login → ignore
        }
    }

    // ❌ DO NOTHING if pendingLogin exists
    // last login without logout is ignored

    return {
        hours: Math.floor(totalSeconds / 3600),
        minutes: Math.floor((totalSeconds % 3600) / 60),
        seconds: totalSeconds % 60,
    };
}

// function calculateLoginTime(records) {
//     let totalLoginTime = 0;
//     let loggedIn = false;
//     let loginTime = null;

//     // Sort by actual timestamp
//     records.sort((a, b) => moment(a.logtime).diff(moment(b.logtime)));

//     records.forEach(record => {
//         const logTime = moment(record.logtime);

//         if (!logTime.isValid()) return; // skip invalid

//         if (record.type == 1) {
//             // LOGIN
//             if (!loggedIn) {
//                 loggedIn = true;
//                 loginTime = logTime;
//             } else {
//                 // Double login → treat as logout+login
//                 const duration = logTime.diff(loginTime, "seconds");
//                 if (duration > 0) totalLoginTime += duration;
//                 loginTime = logTime;
//             }
//         }

//         else if (record.type == 2) {
//             // LOGOUT
//             if (loggedIn) {
//                 const duration = logTime.diff(loginTime, "seconds");
//                 if (duration > 0) totalLoginTime += duration;
//                 loggedIn = false;
//                 loginTime = null;
//             }
//         }
//     });

//     const hours = Math.floor(totalLoginTime / 3600);
//     const minutes = Math.floor((totalLoginTime % 3600) / 60);

//     return { hours, minutes };
// }
// function calculateLoginTime(records, fallbackEndTime = null) {
//     let totalSeconds = 0;

//     // Sort safely
//     records.sort((a, b) =>
//         moment(a.logtime).valueOf() - moment(b.logtime).valueOf()
//     );

//     let pendingLogin = null;

//     for (const record of records) {
//         const time = moment(record.logtime);
//         if (!time.isValid()) continue;

//         if (record.type == 1) {
//             // LOGIN
//             pendingLogin = time;
//         }

//         else if (record.type == 2 && pendingLogin) {
//             // LOGOUT
//             const diff = time.diff(pendingLogin, "seconds");
//             if (diff > 0) totalSeconds += diff;
//             pendingLogin = null;
//         }
//     }

//     // ⏱️ If still logged in, close session using fallback end time
//     if (pendingLogin && fallbackEndTime) {
//         const end = moment(fallbackEndTime);
//         const diff = end.diff(pendingLogin, "seconds");
//         if (diff > 0) totalSeconds += diff;
//     }

//     return {
//         hours: Math.floor(totalSeconds / 3600),
//         minutes: Math.floor((totalSeconds % 3600) / 60),
//     };
// }
exports.getHourlyWages = async (req, res) => {
    try {
        const admins = await Admin.findAll({ where: { role: 'Admin' } })
        res.json({ status: 'success', admins })
    } catch (error) {
        console.error("Add Employee Error:", error);
        return res.status(500).json({
            status: "error",
            message: "Server error",
            details: error.message,
        });
    }
}

exports.getHourlyIndianWages = async (req, res) => {
    try {
        const indianAdmins = [
            "Aiden",
            "Paresh",
            "Sudip",
            "Pradip",
            "Gouri",
            "Archana",
            "Silpa",
        ];

        const admins = await Admin.findAll({
            where: {
                role: "Admin",
                username: {
                    [Op.in]: indianAdmins,
                },
            },
        });

        res.json({
            status: "success",
            admins,
        });
    } catch (error) {
        console.error("Get Hourly Indian Wages Error:", error);
        return res.status(500).json({
            status: "error",
            message: "Server error",
            details: error.message,
        });
    }
};
exports.updateWage = async (req, res) => {
    try {
        const { adminid, wage } = req.body;

        if (!adminid || wage === undefined) {
            return res.json({
                status: "error",
                message: "Admin ID and wage are required",
            });
        }

        const admin = await Admin.findOne({ where: { id: adminid } });

        if (!admin) {
            return res.json({
                status: "error",
                message: "Admin not found",
            });
        }

        const updated = await admin.update({ wage });

        if (updated) {
            return res.json({
                status: "success",
                message: "Wage updated",
                adminid,
                wage,
            });
        } else {
            return res.json({
                status: "error",
                message: "Failed to update wage",
            });
        }

    } catch (error) {
        console.error("Update wage error:", error);
        return res.status(500).json({
            status: "error",
            message: "Server error",
            details: error.message,
        });
    }
};
exports.updateIndianWage = async (req, res) => {
    try {
        const { adminid, wage } = req.body;

        if (!adminid || wage === undefined) {
            return res.json({
                status: "error",
                message: "Admin ID and wage are required",
            });
        }

        const admin = await Admin.findOne({ where: { id: adminid } });

        if (!admin) {
            return res.json({
                status: "error",
                message: "Admin not found",
            });
        }

        const updated = await admin.update({ indian_wage: wage });

        if (updated) {
            return res.json({
                status: "success",
                message: "Wage updated",
                adminid,
                wage,
            });
        } else {
            return res.json({
                status: "error",
                message: "Failed to update wage",
            });
        }

    } catch (error) {
        console.error("Update wage error:", error);
        return res.status(500).json({
            status: "error",
            message: "Server error",
            details: error.message,
        });
    }
};
exports.loginTimes = async (req, res) => {
    try {
        const { cashier, from, to } = req.params;

        if (!cashier || !from || !to) {
            return res.json({
                status: "error",
                message: "cashier, from and to are required",
            });
        }

        const adminlogins = await Logins.findAll({
            where: {
                userid: cashier,
                logtime: {
                    [Op.between]: [from, to],
                },
            },
            order: [["id", "ASC"]],
            // attributes: [
            //     '*',
            //     [Logins.sequelize.literal("logtime"), "converted_logtime"]
            // ]
        });

        const timeArr = adminlogins.map(record => record.toJSON());

        const totalTime = calculateLoginTime(timeArr);

        return res.json({
            status: "success",
            total_time: totalTime,
            login_logouts: timeArr
        });

    } catch (error) {
        console.error("loginTimes error:", error);
        return res.status(500).json({
            status: "error",
            message: "Server error",
            details: error.message
        });
    }
};
exports.weeklyReport = async (req, res) => {
    try {
        // 1️⃣ Fetch all Admin role employees
        const admins = await Admin.findAll({ where: { role: "Admin" } });

        const logins = [];

        const startTime = moment().subtract(1, "week").startOf("day");
        const endTime = moment().endOf("day");

        for (const admin of admins) {
            // 2️⃣ Fetch login times for last week
            const login = await loginTimesHelper(
                admin.id,
                startTime.format("YYYY-MM-DD HH:mm:ss"),
                endTime.format("YYYY-MM-DD HH:mm:ss")
            );

            const time = login.total_time;
            const totalMinutes = time.hours * 60 + time.minutes;

            let hours = time.hours;
            let minutes = time.minutes;

            // ROUNDING RULE:
            if (minutes >= 30) {
                hours++;
            }

            const finalHours = hours;

            const wage = admin.wage;
            const amount = finalHours * wage;

            login.startTime = startTime.format("YYYY-MM-DD HH:mm:ss");
            login.endTime = endTime.format("YYYY-MM-DD HH:mm:ss");
            login.admin_id = admin.id;
            login.email = admin.email;
            login.amount = amount;
            login.username = admin.username;
            login.dateRange =
                startTime.format("YYYY-MM-DD") +
                " - " +
                endTime.format("YYYY-MM-DD");
            login.date = moment().format("YYYY-MM-DD");
            login.rate_per_hour = wage;

            if (!!wage) {
                logins.push(login);
            }
        }

        // 3️⃣ For each login, generate PDF & save it
        for (const login of logins) {
            const data = login;

            // Render HTML template
            const html = await renderTemplate("report3", { data });

            // Generate PDF
            // const browser = await puppeteer.launch({
            //     headless: "new",
            //     args: ["--no-sandbox", "--disable-setuid-sandbox"],
            // });
            const browser = await puppeteer.launch({
                headless: "new",
                args: [
                    "--no-sandbox",
                    "--disable-setuid-sandbox",
                    "--disable-gpu",
                    "--disable-dev-shm-usage",
                    "--single-process",
                    "--no-zygote"
                ],
            });
            const page = await browser.newPage();
            await page.setContent(html, { waitUntil: "networkidle0" });

            const fileName =
                "report_" +
                data.username +
                "_" +
                moment().tz("America/Denver").format("YYYY_MM_DD_HH_mm_ss") +
                ".pdf";

            const savePath = path.join(__dirname, "..", "public", "reports", fileName);

            await page.pdf({
                path: savePath,
                format: "A4",
                printBackground: true,
            });

            await browser.close();

            // 4️⃣ Check reportscron record for that time range
            const check = await ReportsCron.findOne({
                where: {
                    adminid: data.admin_id,
                    date: {
                        [Op.between]: [
                            moment(data.startTime).toDate(),
                            moment(data.endTime).add(1, "hour").toDate(),
                        ],
                    },
                },
                order: [["id", "DESC"]],
            });

            // (You didn’t update or insert anything here — only fetching)
        }

        return res.json({ status: "success", logins });

    } catch (error) {
        console.error("Weekly report error:", error);
        return res.status(500).json({
            status: "error",
            message: "Server error",
            details: error.message,
        });
    }
};
// async function renderTemplate(name, data) {
//     const filePath = path.join(__dirname, "..", "templates", `${name}.html`);
//     let html = fs.readFileSync(filePath, "utf8");

//     // Generate rows for login/logout
//     const rows = data.login_logouts.map(log => `
//         <tr>
//             <td>${log.logtime}</td>
//             <td>${log.type === 1 ? "Login" : "Logout"}</td>
//             <td>${log.msg_response}</td>
//         </tr>
//     `).join("");

//     // Replace placeholders
//     html = html.replace(/{{log_rows}}/g, rows);
//     html = html.replace(/{{total_hours}}/g, data.total_time.hours);
//     html = html.replace(/{{total_minutes}}/g, data.total_time.minutes);
//     html = html.replace(/{{startTime}}/g, data.startTime);
//     html = html.replace(/{{endTime}}/g, data.endTime);
//     html = html.replace(/{{username}}/g, data.username);
//     html = html.replace(/{{admin_id}}/g, data.admin_id);
//     html = html.replace(/{{amount}}/g, data.amount);
//     html = html.replace(/{{rate_per_hour}}/g, data.rate_per_hour);
//     html = html.replace(/{{dateRange}}/g, data.dateRange);
//     html = html.replace(/{{date}}/g, data.date);

//     return html;
// }
async function renderTemplate(name, data) {
    const filePath = path.join(__dirname, "..", "templates", `${name}.html`);
    let html = fs.readFileSync(filePath, "utf8");
    const cssPath = path.join(__dirname, "..", "templates", "pdfstyle.css");
    const css = fs.readFileSync(cssPath, "utf8");

    // INLINE CSS INTO <style>
    html = html.replace('</head>', `<style>${css}</style></head>`);

    const replacements = {
        "{{username}}": data.username || "",
        "{{dateRange}}": data.dateRange || "",
        "{{total_hours}}": data.total_time?.hours || 0,
        "{{total_minutes}}": data.total_time?.minutes || 0,
        "{{rate_per_hour}}": data.rate_per_hour || 0,
        "{{amount}}": data.amount || 0,
        "{{startTime}}": data.startTime || "",
        "{{endTime}}": data.endTime || "",
        "{{date}}": data.date || ""
    };

    // Replace all placeholders
    for (const key in replacements) {
        html = html.replace(new RegExp(key, "g"), replacements[key]);
    }

    return html;
}
async function loginTimesHelper(cashier, from, to) {
    const logs = await Logins.findAll({
        where: {
            userid: cashier,
            logtime: {
                [Op.between]: [from, to],
            },
        },
        order: [["id", "ASC"]],
        // attributes: [
        //     "*",
        //     [Logins.sequelize.literal("logtime"), "converted_logtime"],
        // ],
    });

    const arr = logs.map((l) => l.toJSON());

    const total_time = calculateLoginTime(arr);

    return {
        total_time,
        login_logouts: arr,
    };
}
// function calculateIndianDaysWithHalf(logins) {
//     const dayMap = {};

//     logins.forEach(l => {
//         if (!l.logtime) return;

//         const date = moment(l.logtime).format("YYYY-MM-DD");

//         if (!dayMap[date]) {
//             dayMap[date] = [];
//         }

//         dayMap[date].push(moment(l.logtime));
//     });

//     let totalDays = 0;

//     Object.values(dayMap).forEach(times => {
//         // total hours for that day
//         let dayHours = 0;

//         for (let i = 0; i < times.length; i += 2) {
//             const login = times[i];
//             const logout = times[i + 1];

//             if (login && logout) {
//                 dayHours += moment.duration(logout.diff(login)).asHours();
//             }
//         }

//         // Half / Full day rule
//         // if (dayHours > 0 && dayHours < 7.5) {
//         //     totalDays += 0.5;
//         // } else if (dayHours >= 8) {
//         totalDays += 1;
//         // } else if (dayHours >= 8.5) {
//         //     totalDays += 1.5;
//         // }
//     });

//     return totalDays;
// }
function calculateIndianDaysFromHours(logins) {
    let totalHours = 0;

    for (let i = 0; i < logins.length; i += 2) {
        const login = moment(logins[i].logtime);
        const logout = logins[i + 1]
            ? moment(logins[i + 1].logtime)
            : null;

        if (logout) {
            totalHours += moment
                .duration(logout.diff(login))
                .asHours();
        }
    }

    return totalHours / 8;
}
function calculateIndianDaysWithHalf(logins) {
    const dayMap = {};

    // 1️⃣ Group login/logout times by date
    for (let i = 0; i < logins.length; i += 2) {
        const login = logins[i];
        const logout = logins[i + 1];

        if (!login || !logout) continue;

        const loginTime = moment(login.logtime);
        const logoutTime = moment(logout.logtime);

        const dateKey = loginTime.format("YYYY-MM-DD");

        if (!dayMap[dateKey]) {
            dayMap[dateKey] = 0;
        }

        const hours = moment
            .duration(logoutTime.diff(loginTime))
            .asHours();

        dayMap[dateKey] += hours;
    }

    // 2️⃣ Apply Indian wage day rules
    let totalDays = 0;

    Object.values(dayMap).forEach(dayHours => {
        if (dayHours > 0 && dayHours < 7.5) {
            totalDays += 0.5;
        } else if (dayHours > 8.5) {
            totalDays += 1.5;
        } else {
            totalDays += 1;
        }
    });

    return totalDays;
}
exports.report = async (req, res) => {
    try {
        const authHeader = req.headers['authorization'] || req.headers['Authorization'];
        if (!authHeader) {

            return res.json({
                status: 'error',
                message: 'Authorization header is missing',
            });
        }

        const token = authHeader.split(' ')[1];
        if (!token) {

            return res.json({
                status: 'error',
                message: 'Token is missing',
            });
        }

        const decoded = jwt.verify(token, process.env.JWT_SECRET);
        const userId = decoded.id;
        const user = await Admin.findOne({ where: { id: userId } });

        if (user.role !== "Super") {
            return res.status(401).json({
                status: "error",
                message: "Unauthorized",
            });
        }

        const admins = await Admin.findAll();

        const { from, to, admin, timezone } = req.query;
        let tz = "Asia/Kolkata"; // default IST

        console.log(timezone)
        if (timezone && timezone.toUpperCase() === "MST") {
            tz = "America/Denver";
        }
        // If no filters → return only admin list
        if (!from || !to || !admin) {
            return res.json({
                status: "success",
                // admins,
                data: [],
                total_time: null,
            });
        }

        // Find selected admin
        const selectedAdmin = await Admin.findOne({
            where: { username: admin },
        });

        if (!selectedAdmin) {
            return res.json({
                status: "error",
                message: "Admin not found",
            });
        }

        // Convert dates
        let startDate, endDate;

        if (!from && !to) {
            startDate = moment().startOf("day");
            endDate = moment().endOf("day");
        } else {
            startDate = moment(from);
            endDate = moment(to);
        }

        // Fetch schedules
        // const schedules = await Schedule.findAll({
        //     where: {
        //         adminid: selectedAdmin.id,
        //         end_time: {
        //             [Op.between]: [
        //                 startDate.format("YYYY-MM-DD HH:mm:ss"),
        //                 endDate.format("YYYY-MM-DD HH:mm:ss"),
        //             ],
        //         },
        //     },
        //     include: [{ model: Admin, as: "admin", attributes: { exclude: ['password'] } }],
        // });

        // If schedules found → use schedule range
        // if (schedules.length > 0) {
        //     const firstSchedule = schedules[0];
        //     const lastSchedule = schedules[schedules.length - 1];

        //     const from_date = moment(firstSchedule.start_time).format("YYYY-MM-DD HH:mm:ss");
        //     const to_date = moment(lastSchedule.end_time).format("YYYY-MM-DD HH:mm:ss");

        //     const adminlogins = await Logins.findAll({
        //         where: {
        //             userid: selectedAdmin.id,
        //             logtime: {
        //                 [Op.between]: [from_date, to_date],
        //             },
        //         },
        //         order: [["id", "ASC"]],
        //         attributes: [
        //             "*",
        //             [Logins.sequelize.literal("logtime"), "converted_logtime"],
        //         ],
        //         include: [{ model: Admin, as: "admin", attributes: { exclude: ['password'] } }],
        //     });

        //     const timeArr = adminlogins.map(a => a.toJSON());
        //     const total_time = calculateLoginTime(timeArr);

        //     return res.json({
        //         status: "success",
        //         admins,
        //         cashier: admin,
        //         total_time,
        //         data: timeArr,
        //     });
        // }

        // If no schedules → use direct range logic

        const reqTz =
            timezone === "MST"
                ? "America/Denver"
                : "Asia/Kolkata";

        // const from_date = moment.tz(from, "YYYY-MM-DD HH:mm:ss", tz)
        //     .format("YYYY-MM-DD HH:mm:ss");

        // const to_date = moment.tz(to, "YYYY-MM-DD HH:mm:ss", tz)
        //     .format("YYYY-MM-DD HH:mm:ss");
        const from_date = moment
            .tz(from, "YYYY-MM-DD HH:mm:ss", reqTz) // parse as user TZ
            .utc()                                 // 🔥 convert to UTC
            .format("YYYY-MM-DD HH:mm:ss");

        const to_date = moment
            .tz(to, "YYYY-MM-DD HH:mm:ss", reqTz)
            .utc()
            .format("YYYY-MM-DD HH:mm:ss");
        // const from_date = moment(from).format("YYYY-MM-DD HH:mm:ss");
        // const to_date = moment(to).format("YYYY-MM-DD HH:mm:ss");

        const adminlogins = await Logins.findAll({
            where: {
                userid: selectedAdmin.id,
                created_at: {
                    [Op.between]: [from_date, to_date],
                },
            },
            order: [["id", "ASC"]],
            attributes: [
                "id",
                "userid",
                "type",
                "logtime",
                "msg_response",
                "created_at",
                "updated_at",
                ["id", "real_id"]  // example
            ]
            // attributes: [
            //     "*",
            //     [Logins.sequelize.literal("logtime"), "logtime"],
            // ],
            // include: [{ model: Admin, as: "admin" }],
        });

        // const timeArr = adminlogins.map(a => a.toJSON());
        // const timeArr = adminlogins.map(a => {
        //     const row = a.toJSON();
        //     row.logtime = moment.utc(row.logtime).tz(tz).format("YYYY-MM-DD HH:mm:ss");
        //     return row;
        // });
        const timeArr = adminlogins.map(a => {
            const row = a.toJSON();
            // keep as UTC moment-compatible value
            row.logtime = moment.utc(row.logtime);
            return row;
        });
        // ================================
        // OLD SYSTEM (HOUR BASED)
        // ================================
        // const total_time = calculateLoginTime(timeArr);
        const total_time = calculateLoginTime(timeArr);
        let { hours, minutes } = total_time;

        // Round up if 30+ minutes
        let roundedHours = minutes >= 30 ? hours + 1 : hours;

        const wage = selectedAdmin.wage;
        let amount = roundedHours * wage;
        let isIndianWageApplied = false;
        // ================================
        // 🇮🇳 INDIAN WAGE (DAY BASED)
        // ================================
        const indian_wage = selectedAdmin.indian_wage;

        if (indian_wage && indian_wage > 0) {
            const workedDays = calculateIndianDaysWithHalf(timeArr);
            amount = workedDays * indian_wage;
            isIndianWageApplied = true;
        }
        amount = Number(amount.toFixed(2));
        return res.json({
            status: "success",
            cashier: admin,
            total_time,
            data: timeArr,
            amount,
            cashier_id: selectedAdmin.id,
            isIndianWageApplied,
            timezone: tz
        });

    } catch (error) {
        console.error("Report Error:", error);
        return res.status(500).json({
            status: "error",
            message: "Server error",
            details: error.message,
        });
    }
};
exports.exportReport = async (req, res) => {
    try {
        const { admin: adminUsername, from, to } = req.body;

        if (!adminUsername || !from || !to) {
            return res.json({ status: "error", message: "Missing required fields" });
        }

        const admin = await Admin.findOne({ where: { username: adminUsername } });

        if (!admin) {
            return res.json({ status: "error", message: "Admin not found" });
        }

        const fromDate = moment(from);
        const toDate = moment(to);

        const schedules = await Schedule.findAll({
            where: {
                adminid: admin.id,
                end_time: {
                    [Op.between]: [
                        fromDate.format("YYYY-MM-DD HH:mm:ss"),
                        toDate.format("YYYY-MM-DD HH:mm:ss")
                    ]
                }
            },
            order: [["id", "ASC"]],
        });

        let startTime, endTime;

        if (schedules.length > 0) {
            startTime = schedules[0].start_time;
            endTime = schedules[schedules.length - 1].end_time;
        } else {
            startTime = fromDate.format("YYYY-MM-DD HH:mm:ss");
            endTime = toDate.format("YYYY-MM-DD HH:mm:ss");
        }

        // Fetch login times
        const loginData = await loginTimesHelper(admin.id, startTime, endTime);
        const time = loginData.total_time;

        let hours = time.hours;
        let minutes = time.minutes;

        if (minutes >= 30) hours++; // rounding rule

        const amount = hours * admin.wage;

        const payload = {
            ...loginData,
            startTime,
            endTime,
            admin_id: admin.id,
            email: admin.email,
            username: admin.username,
            rate_per_hour: admin.wage,
            amount,
            dateRange: `${fromDate.format("YYYY-MM-DD")} - ${toDate.format("YYYY-MM-DD")}`,
            date: moment().format("YYYY-MM-DD"),
        };

        // Render HTML
        const html = await renderTemplate("report3", payload);

        // Prepare file info
        const fileName =
            `report_${payload.username}_` +
            moment().tz("America/Denver").format("YYYY_MM_DD_HH_mm_ss") +
            `.pdf`;

        const reportsDir = path.join(__dirname, "..", "public", "reports");
        if (!fs.existsSync(reportsDir)) {
            fs.mkdirSync(reportsDir, { recursive: true });
        }

        const savePath = path.join(reportsDir, fileName);

        // Generate PDF using html-pdf-node
        let pdfResult = await html_to_pdf.generatePdf(
            { content: html },
            {
                width: "210mm",     // A4 width
                height: "130mm",    // custom small height
                printBackground: true,
                margin: {
                    top: "5mm",
                    bottom: "5mm",
                    left: "10mm",
                    right: "10mm",
                },
            }
        );

        // Some versions return a raw buffer, some return { buffer: <Buffer> }
        const pdfBuffer = Buffer.isBuffer(pdfResult)
            ? pdfResult
            : pdfResult.buffer;

        // Save PDF to disk
        fs.writeFileSync(savePath, pdfBuffer);

        // Build public URL
        const pdfUrl = `${process.env.APP_URL}/reports/${fileName}`;

        return res.json({
            status: "success",
            url: pdfUrl,
        });

    } catch (error) {
        console.error("Export Report Error:", error);
        return res.status(500).json({
            status: "error",
            message: "Server error",
            details: error.message,
        });
    }
};
// exports.exportReport = async (req, res) => {
//     try {
//         const { admin: adminUsername, from, to } = req.body;

//         if (!adminUsername || !from || !to) {
//             return res.json({ status: "error", message: "Missing required fields" });
//         }

//         const admin = await Admin.findOne({ where: { username: adminUsername } });

//         if (!admin) {
//             return res.json({ status: "error", message: "Admin not found" });
//         }

//         const fromDate = moment(from);
//         const toDate = moment(to);

//         const schedules = await Schedule.findAll({
//             where: {
//                 adminid: admin.id,
//                 end_time: {
//                     [Op.between]: [
//                         fromDate.format("YYYY-MM-DD HH:mm:ss"),
//                         toDate.format("YYYY-MM-DD HH:mm:ss")
//                     ]
//                 }
//             },
//             order: [["id", "ASC"]],
//             // include: [{ model: Admin, as: "admin" }]
//         });

//         let startTime, endTime;

//         // If schedules exist → derive time range from schedules
//         if (schedules.length > 0) {
//             startTime = schedules[0].start_time;
//             endTime = schedules[schedules.length - 1].end_time;
//         } else {
//             // Otherwise use direct range
//             startTime = fromDate.format("YYYY-MM-DD HH:mm:ss");
//             endTime = toDate.format("YYYY-MM-DD HH:mm:ss");
//         }

//         // Fetch login times for selected window
//         const loginData = await loginTimesHelper(admin.id, startTime, endTime);
//         const time = loginData.total_time;

//         let hours = time.hours;
//         let minutes = time.minutes;

//         // Round hours
//         if (minutes >= 30) hours++;

//         const finalHours = hours;
//         const wage = admin.wage;
//         const amount = finalHours * wage;

//         const payload = {
//             ...loginData,
//             startTime,
//             endTime,
//             admin_id: admin.id,
//             email: admin.email,
//             username: admin.username,
//             rate_per_hour: admin.wage,
//             amount,
//             dateRange:
//                 moment(fromDate).format("YYYY-MM-DD") +
//                 " - " +
//                 moment(toDate).format("YYYY-MM-DD"),
//             date: moment().format("YYYY-MM-DD")
//         };

//         // Render HTML
//         const html = await renderTemplate("report3", payload);

//         // Generate PDF with Puppeteer
//         const browser = await puppeteer.launch({
//             headless: "new",
//             args: ["--no-sandbox", "--disable-setuid-sandbox"]
//         });

//         const page = await browser.newPage();
//         await page.setContent(html, { waitUntil: "networkidle0" });

//         const fileName =
//             `report_${payload.username}_` +
//             moment().tz("America/Denver").format("YYYY_MM_DD_HH_mm_ss") +
//             `.pdf`;

//         // Generate PDF
//         // const pdfBuffer = await page.pdf({
//         //     format: "A4",
//         //     printBackground: true
//         // });
//         const pdfBuffer = await page.pdf({
//             width: "210mm",
//             height: "110mm", // shorter height
//             printBackground: true
//         });
//         await browser.close();

//         // SAVE FILE ON SERVER
//         const savePath = path.join(__dirname, "..", "public", "reports", fileName);
//         fs.writeFileSync(savePath, pdfBuffer);

//         // // DOWNLOAD TO CLIENT
//         // res.setHeader("Content-Type", "application/pdf");
//         // res.setHeader("Content-Disposition", `attachment; filename="${fileName}"`);
//         // return res.end(Buffer.from(pdfBuffer));
//         // MAKE PUBLIC URL
//         const pdfUrl = `${process.env.APP_URL}/reports/${fileName}`;

//         return res.json({
//             status: "success",
//             url: pdfUrl
//         });

//     } catch (error) {
//         console.error("Export Report Error:", error);
//         return res.status(500).json({
//             status: "error",
//             message: "Server error",
//             details: error.message
//         });
//     }
// };
exports.addBalance = async (req, res) => {
    try {
        const { id, amount } = req.body;

        if (!id || !amount) {
            return res.status(400).json({
                status: "error",
                message: "id and amount are required"
            });
        }

        // Find cashier/admin
        const admin = await Admin.findOne({ where: { id } });

        if (!admin) {
            return res.status(404).json({
                status: "error",
                message: "Admin not found"
            });
        }
        await Transactions.create({
            admin_id: admin.id,
            type: '1',
            amount: amount,
            method: 'Balance Add',
            status: '1',
            request: JSON.stringify(req.body)

        })
        // Increment balance
        await admin.increment("balance", { by: amount });

        return res.json({
            status: "success",
            message: "Employee balance updated successfully",
            admin
        });

    } catch (error) {
        console.error("Add Balance Error:", error);
        return res.status(500).json({
            status: "error",
            message: "Server error",
            details: error.message
        });
    }
};
exports.quickAddBalance = async (req, res) => {
    try {
        const { id, amount } = req.body;

        if (!id || !amount) {
            return res.status(400).json({
                status: "error",
                message: "id and amount are required"
            });
        }

        // Find cashier/admin
        const admin = await Admin.findOne({ where: { id } });

        if (!admin) {
            return res.status(404).json({
                status: "error",
                message: "Admin not found"
            });
        }
        await Transactions.create({
            admin_id: admin.id,
            type: '1',
            amount: amount,
            method: 'Quick Add',
            status: '1',
            request: JSON.stringify(req.body)
        })
        // Increment balance
        await admin.increment("balance", { by: amount });

        return res.json({
            status: "success",
            message: "Employee balance updated successfully",
            admin
        });

    } catch (error) {
        console.error("Add Balance Error:", error);
        return res.status(500).json({
            status: "error",
            message: "Server error",
            details: error.message
        });
    }
};
exports.removeBalance = async (req, res) => {
    try {
        const { id, amount } = req.body;

        if (!id || !amount) {
            return res.status(400).json({
                status: "error",
                message: "id and amount are required"
            });
        }

        // Find cashier/admin
        const admin = await Admin.findOne({ where: { id } });

        if (!admin) {
            return res.status(404).json({
                status: "error",
                message: "Admin not found"
            });
        }
        await Transactions.create({
            admin_id: admin.id,
            type: '2',
            amount: amount,
            method: 'Balance Removed',
            status: '1',
            request: JSON.stringify(req.body)

        })
        // Increment balance
        await admin.decrement("balance", { by: amount });

        return res.json({
            status: "success",
            message: "Employee balance updated successfully",
            admin
        });

    } catch (error) {
        console.error("Add Balance Error:", error);
        return res.status(500).json({
            status: "error",
            message: "Server error",
            details: error.message
        });
    }
};
exports.wagesCharts = async (req, res) => {
    try {
        let fromDate, toDate;

        if (req.body.from_date && req.body.to_date) {
            fromDate = moment(req.body.from_date).format("YYYY-MM-DD") + " 00:00:00";
            toDate = moment(req.body.to_date).format("YYYY-MM-DD") + " 23:59:00";
        } else {
            fromDate = moment().subtract(7, "days").format("YYYY-MM-DD") + " 00:00:00";
            toDate = moment().format("YYYY-MM-DD") + " 23:59:00";
        }

        const admins = await Admin.findAll();
        const logins = [];

        for (const admin of admins) {
            const loginData = await loginTimesHelper(admin.id, fromDate, toDate);
            const time = loginData.total_time;

            const finalHours = time.minutes >= 30 ? time.hours + 1 : time.hours;
            const amount = finalHours * admin.wage;

            if (admin.wage) {
                logins.push({
                    username: admin.username,
                    amount,
                    date: moment(fromDate).format("YYYY-MM-DD") + " - " + moment(toDate).format("YYYY-MM-DD")
                });
            }
        }

        const labels = logins.map((item) => item.username);
        const values = logins.map((item) => item.amount);

        return res.json({
            status: "success",
            labels,
            values,
            total: values.reduce((a, b) => a + b, 0)
        });

    } catch (err) {
        console.error(err);
        return res.status(500).json({
            status: "error",
            message: err.message
        });
    }
};
// exports.wagesCharts = async (req, res) => {
//     try {
//         let fromDate, toDate;

//         // If custom dates provided
//         if (req.body.from_date && req.body.to_date) {
//             fromDate = moment(req.body.from_date).format("YYYY-MM-DD") + " 00:00:00";
//             toDate = moment(req.body.to_date).format("YYYY-MM-DD") + " 23:59:00";
//         } else {
//             // Default: last week
//             fromDate = moment().subtract(7, "days").format("YYYY-MM-DD") + " 00:00:00";
//             toDate = moment().format("YYYY-MM-DD") + " 23:59:00";
//         }

//         const admins = await Admin.findAll();
//         const logins = [];

//         for (const admin of admins) {
//             // Fetch login time data using your helper
//             const loginData = await loginTimesHelper(admin.id, fromDate, toDate);

//             const time = loginData.total_time; // { hours, minutes }
//             const totalMinutes = (time.hours * 60) + time.minutes;

//             let hours = time.hours;
//             let minutes = time.minutes;

//             // Laravel rounding rule: if minutes >= 30 → round UP 1 hour
//             const finalTotal = minutes >= 30 ? hours + 1 : hours;

//             const wage = admin.wage;
//             const amount = finalTotal * wage;

//             const loginDetails = {
//                 startTime: fromDate,
//                 endTime: toDate,
//                 admin_id: admin.id,
//                 email: admin.email,
//                 amount,
//                 username: admin.username,
//                 dateRange:
//                     moment(fromDate).format("YYYY-MM-DD") +
//                     " - " +
//                     moment(toDate).format("YYYY-MM-DD"),
//                 date: moment().format("YYYY-MM-DD"),
//                 rate_per_hour: admin.wage
//             };

//             if (admin.wage) {
//                 logins.push(loginDetails);
//             }
//         }

//         const totalAmount = logins.reduce((sum, item) => sum + item.amount, 0);

//         return res.json({
//             logins,
//             total: totalAmount
//         });

//     } catch (err) {
//         console.error("Wages chart error:", err);
//         return res.status(500).json({
//             status: "error",
//             message: "Server error",
//             details: err.message
//         });
//     }
// };
exports.registerCoinFlowUser = async (req, res) => {
    try {
        const authHeader = req.headers['authorization'] || req.headers['Authorization'];
        if (!authHeader) {

            return res.json({
                status: 'error',
                message: 'Authorization header is missing',
            });
        }

        const token = authHeader.split(' ')[1];
        if (!token) {

            return res.json({
                status: 'error',
                message: 'Token is missing',
            });
        }

        const decoded = jwt.verify(token, process.env.JWT_SECRET);
        const userId = decoded.id;

        // const userAuth = req.user; // from JWT middleware or session

        // Destructure request body
        const {
            physicalAddress,
            city,
            state,
            zip,
            dob,
            ssn,
            method // optional
        } = req.body;

        // SIMPLE REQUIRED FIELD CHECKS
        if (!physicalAddress) return res.json({ status: "error", message: "Physical Address is required" });
        if (!city) return res.json({ status: "error", message: "City is required" });
        if (!state) return res.json({ status: "error", message: "State is required" });
        if (!zip) return res.json({ status: "error", message: "Zip is required" });
        if (!dob) return res.json({ status: "error", message: "DOB is required" });
        if (!ssn) return res.json({ status: "error", message: "SSN is required" });

        // Optional field validation
        // if (method && !["bank", "paypal", "venmo"].includes(method)) {
        //     return res.json({ status: "error", message: "Invalid payment method" });
        // }

        // Find user
        const user = await Admin.findOne({ where: { id: userId } });

        if (!user) {
            return res.status(404).json({
                status: "error",
                message: "User not found"
            });
        }

        // CoinFlow payload
        const data = {
            email: user.email,
            firstName: user.first,
            surName: user.last,
            physicalAddress,
            city,
            state,
            zip,
            country: "US",
            dob,
            ssn
        };

        // Headers
        const headers = {
            "Authorization": "coinflow_prod_fb228961487143f6867d324ff6e58dd7_49d72e1213644a06becae93b36a36fad",
            "accept": "application/json",
            "content-type": "application/json",
            "x-coinflow-auth-user-id": String(user.id)
        };

        // API Request
        const response = await axios.post(
            "https://api.coinflow.cash/api/withdraw/kyc/attested",
            data,
            { headers }
        );

        const responseData = response.data;
        const status = responseData?.withdrawer?.verification?.status;

        // APPROVED OR PARTIAL
        if (["approved", "partial-approval"].includes(status)) {
            await user.update({ cfAccount: "1" });

            return res.json({
                status: "success",
                message: "CoinFlow Registration Successful",
                data: responseData,
                redirectUrl: "/dashboard"
            });
        }
        else if (
            res.details &&
            typeof res.details === "string" &&
            res.details.includes("KYC verification already exists")
        ) {
            await user.update({ cfAccount: "1" });

            return res.json({
                status: "success",
                message: "KYC already exists"
            });
        }
        // PENDING
        return res.json({
            status: "pending",
            message: "CoinFlow Registration is under process",
            data: responseData
        });

    } catch (err) {
        const apiError = err.response?.data || { message: err.message };
        const authHeader = req.headers['authorization'] || req.headers['Authorization'];
        if (!authHeader) {

            return res.json({
                status: 'error',
                message: 'Authorization header is missing',
            });
        }

        const token = authHeader.split(' ')[1];
        if (!token) {

            return res.json({
                status: 'error',
                message: 'Token is missing',
            });
        }

        const decoded = jwt.verify(token, process.env.JWT_SECRET);
        const userId = decoded.id;
        // KYC ALREADY EXISTS
        if (
            apiError.details &&
            typeof apiError.details === "string" &&
            apiError.details.includes("KYC verification already exists")
        ) {
            const user = await Admin.findOne({ where: { id: userId } });

            await user.update({ cfAccount: "1" });

            return res.json({
                status: "success",
                message: "KYC already exists"
            });
        }

        return res.status(500).json({
            status: "error",
            message: "CoinFlow KYC request failed",
            details: apiError
        });
    }
};
async function getCoinFlowWithdrawer(userId) {
    try {
        const headers = {
            "Authorization": "coinflow_prod_fb228961487143f6867d324ff6e58dd7_49d72e1213644a06becae93b36a36fad",
            "accept": "application/json",
            "x-coinflow-auth-user-id": String(userId)
        };

        const response = await axios.get(
            "https://api.coinflow.cash/api/withdraw",
            { headers }
        );

        return response.data;

    } catch (err) {
        // Return API error if available
        return err.response?.data || { error: err.message };
    }
}
exports.getCoinFlowRedirectUrl = async (req, res) => {
    try {
        const { userId, site } = req.body;
        const merchantId = "Rcamusement";

        if (!userId) {
            return res.status(400).json({ error: "userId is required" });
        }

        // Get session key from CoinFlow API
        const sessionKey = await getSessionKey(userId);

        if (!sessionKey || !sessionKey.key) {
            return res.status(500).json({
                error: "Failed to generate session key"
            });
        }

        const redirectUrl = encodeURIComponent(site);
        const baseUrl = "https://coinflow.cash"; // production

        const coinflowUrl =
            `${baseUrl}/solana/withdraw/${merchantId}` +
            `?sessionKey=${sessionKey.key}&bankAccountLinkRedirect=${redirectUrl}`;

        return res.json({
            status: "success",
            redirectUrl: coinflowUrl
        });

    } catch (err) {
        console.error("Error generating CoinFlow redirect URL:", err.message);
        return res.status(500).json({
            error: "Internal server error"
        });
    }
};
async function getSessionKey(userId) {
    try {
        const baseUrl = "https://api.coinflow.cash/api/auth/session-key";

        const authToken =
            "coinflow_prod_fb228961487143f6867d324ff6e58dd7_49d72e1213644a06becae93b36a36fad";

        const headers = {
            Authorization: authToken,
            accept: "application/json",
            "x-coinflow-auth-user-id": String(userId)
        };

        const response = await axios.get(baseUrl, { headers });

        return response.data; // same as $response->json()

    } catch (error) {
        console.error("Error fetching CoinFlow session key:", error.response?.data || error.message);
        return null;
    }
}
// exports.storeCoinFlowBankToken = async (req, res) => {
//     try {
//         const { userId } = req.body;

//         // Validate input
//         if (!userId) {
//             return res.status(400).json({
//                 status: "error",
//                 message: "UserId is required"
//             });
//         }

//         // Fetch user
//         const user = await Admin.findOne({ where: { id: userId } });

//         if (!user) {
//             return res.status(404).json({
//                 status: "error",
//                 message: "User not found"
//             });
//         }

//         // Fetch CoinFlow withdrawer info
//         const userInfo = await getCoinFlowWithdrawer(userId);

//         if (!userInfo || !userInfo.withdrawer) {
//             return res.status(404).json({
//                 status: "error",
//                 message: "Withdrawer not found"
//             });
//         }

//         const bankAccounts = userInfo.withdrawer.bankAccounts || [];

//         // Get last bank account
//         let lastBankAccount = null;
//         if (bankAccounts.length > 0) {
//             lastBankAccount = bankAccounts[bankAccounts.length - 1];
//         }

//         // Save token if exists
//         if (lastBankAccount && lastBankAccount.token) {
//             await user.update({ cfAccountToken: lastBankAccount.token });
//         }

//         return res.json({
//             status: "success",
//             bankAccount: lastBankAccount
//         });

//     } catch (err) {
//         console.error("CoinFlow storeCoinFlowBankToken error:", err);

//         return res.status(500).json({
//             error: "CoinFlow request failed",
//             details: err.response?.data || err.message
//         });
//     }
// };
exports.storeCoinFlowBankToken = async (req, res) => {
    try {
        const { userId } = req.body;
        if (!userId) {
            return res.json({ status: 'error', message: "UserId is required" });
        }

        const user = await Admin.findOne({ where: { id: userId } });
        if (!user) {
            return res.json({ status: 'error', message: "User not found" });
        }

        const userInfo = await getCoinFlowWithdrawer(userId.toString());
        if (!userInfo?.withdrawer) {
            return res.json({ status: "error", message: "Withdrawer not found" });
        }

        const bankAccounts = userInfo.withdrawer.bankAccounts || [];
        if (bankAccounts.length === 0) {
            return res.json({ status: "success", message: "No bank accounts found" });
        }

        let insertedCount = 0;
        for (const bank of bankAccounts) {
            const { token, alias, last4, routingNumber } = bank;

            if (!token) continue;

            // Check if this account already exists for this user
            const exists = await LinkedAccounts.findOne({
                where: { adminid: userId, account_token: token }
            });

            if (!exists) {
                // Insert new bank account
                await LinkedAccounts.create({
                    adminid: userId,
                    account_token: token,
                    alias: alias || null,
                    last4: last4 || null,
                    routing_number: routingNumber || null
                });
                insertedCount++;
            }
        }

        // Update user's latest cfAccountToken (last in array)
        const lastBankAccount = bankAccounts[bankAccounts.length - 1];
        if (lastBankAccount?.token) {
            user.cfAccountToken = lastBankAccount.token;
            await user.save();
        }

        return res.json({
            status: "success",
            message: `${insertedCount} new bank account(s) added`,
            totalAccounts: bankAccounts.length,
        });

    } catch (error) {
        console.error("CoinFlow Bank Sync Error:", error.response?.data || error.message);
        return res.status(500).json({
            status: "error",
            message: "CoinFlow bank account sync failed",
            details: error.response?.data || error.message,
        });
    }
};
exports.getCFBankAccounts = async (req, res) => {
    const { userid } = req.body;

    const authHeader = req.headers['authorization'] || req.headers['Authorization'];
    if (!authHeader) {

        return res.json({ status: 'error', message: 'Authorization header is missing' });
    }

    const token = authHeader.split(' ')[1];
    if (!token) {

        return res.json({ status: 'error', message: 'Token is missing' });
    }

    try {
        // Verify JWT token
        const decoded = jwt.verify(token, process.env.JWT_SECRET);
        const userIdFromToken = decoded.id;

        if (!userid || userIdFromToken.toString() !== userid.toString()) {

            return res.json({ status: 'error', message: 'Invalid user ID or token mismatch' });
        }

        // Fetch user and cards
        const user = await Admin.findByPk(userid);
        if (!user) {

            return res.json({ status: 'error', message: 'User not found' });
        }

        const accounts = await LinkedAccounts.findAll({ where: { adminid: userid } });

        if (accounts.length === 0) {

            return res.json({ status: 'success', accounts: [] });
        }

        const accountData = accounts.map(account => ({
            id: account.id,
            alias: account.alias,
            token: account.account_token,
            routingNumber: account.routing_number,
            last4: account.last4

        }));

        return res.json({ status: 'success', accounts: accountData });
    } catch (error) {

        return res.json({ status: 'error', message: 'Server error', error: error.message });
    }
};
exports.registerPaypalToken = async (req, res) => {
    try {
        const { userId, paypal } = req.body;
        // const userId = req.user.id;              // from JWT/session auth
        // const paypal = req.body.paypal;          // phone number string

        if (!paypal) {
            return res.json({ status: "error", message: "PayPal phone number is required" });
        }

        // Get user from database
        const user = await Admin.findOne({ where: { id: userId } });

        if (!user) {
            return res.json({
                status: "error",
                message: "User not found"
            });
        }

        // CoinFlow API headers
        const headers = {
            Authorization: "coinflow_prod_fb228961487143f6867d324ff6e58dd7_49d72e1213644a06becae93b36a36fad",
            accept: "application/json",
            "x-coinflow-auth-user-id": String(userId)
        };

        // 🔹 POST request to CoinFlow to set PayPal account
        const response = await axios.post(
            "https://api.coinflow.cash/api/withdraw/paypal",
            { phoneNumber: String(paypal) },
            { headers }
        );

        // Validate response
        const data = response.data;

        // 🔹 Fetch updated CoinFlow withdrawer info
        const userInfo = await getCoinFlowWithdrawer(userId);

        if (!userInfo || !userInfo.withdrawer) {
            return res.json({
                status: "error",
                message: "Unable to link PayPal"
            });
        }

        // Extract PayPal details
        const paypalAccount = userInfo.withdrawer.paypal || null;

        if (paypalAccount && paypalAccount.token) {
            // Save token to database
            await user.update({ cfPaypalToken: paypalAccount.token });

            return res.json({
                status: "success",
                message: "PayPal link successful",
                paypal: paypalAccount
            });
        }

        return res.json({
            status: "error",
            message: "Unable to link PayPal"
        });

    } catch (err) {
        console.error("PayPal link error:", err.response?.data || err.message);

        return res.json({
            status: "error",
            message: "Unable to link PayPal",
            details: err.response?.data || err.message
        });
    }
};
exports.registerVenmoToken = async (req, res) => {
    try {
        const { userId, venmo } = req.body;
        // const userId = req.user.id;      // authenticated user id
        // const venmo = req.body.venmo;    // venmo phone number

        if (!venmo) {
            return res.json({
                status: "error",
                message: "Venmo phone number is required"
            });
        }

        const user = await Admin.findOne({ where: { id: userId } });

        if (!user) {
            return res.json({
                status: "error",
                message: "User not found"
            });
        }

        // CoinFlow headers
        const headers = {
            Authorization: "coinflow_prod_fb228961487143f6867d324ff6e58dd7_49d72e1213644a06becae93b36a36fad",
            accept: "application/json",
            "x-coinflow-auth-user-id": String(userId)
        };

        // 🔹 Step 3: POST to CoinFlow Venmo endpoint
        const response = await axios.post(
            "https://api.coinflow.cash/api/withdraw/venmo",
            { phoneNumber: String(venmo) },
            { headers }
        );

        // Validate response
        const data = response.data;

        // 🔹 Step 5: Fetch updated withdrawer info
        const userInfo = await getCoinFlowWithdrawer(userId);

        if (!userInfo || !userInfo.withdrawer) {
            return res.json({
                status: "error",
                message: "Unable to link Venmo"
            });
        }

        // 🔹 Step 6: Extract Venmo details
        const venmoAccount = userInfo.withdrawer.venmo || null;

        if (venmoAccount && venmoAccount.token) {
            // Save token to DB
            await user.update({ cfVenmoToken: venmoAccount.token });

            return res.json({
                status: "success",
                message: "Venmo link successful",
                venmo: venmoAccount
            });
        }

        return res.json({
            status: "error",
            message: "Unable to link Venmo"
        });

    } catch (err) {
        console.error("Venmo link error:", err.response?.data || err.message);

        return res.json({
            status: "error",
            message: "Unable to link Venmo",
            details: err.response?.data || err.message
        });
    }
};
exports.requestWithdraw = async (req, res) => {
    let userId = null;
    let amount = 0;
    let method = 'bank';

    const sequelize = Admin.sequelize;
    const t = await sequelize.transaction();

    try {
        /* =======================
           AUTH
        ======================= */
        const authHeader = req.headers['authorization'] || req.headers['Authorization'];
        if (!authHeader) {
            await t.rollback();
            return res.json({ status: 'error', message: 'Authorization header is missing' });
        }

        const token = authHeader.split(' ')[1];
        if (!token) {
            await t.rollback();
            return res.json({ status: 'error', message: 'Token is missing' });
        }

        const decoded = jwt.verify(token, process.env.JWT_SECRET);
        userId = decoded.id;

        /* =======================
           🔒 LOCK ADMIN ROW
        ======================= */
        const admin = await Admin.findOne({
            where: { id: userId },
            transaction: t,
            lock: t.LOCK.UPDATE   // 🔥 SELECT ... FOR UPDATE
        });

        if (!admin) {
            await t.rollback();
            return res.json({ status: 'error', message: "User not found!" });
        }

        /* =======================
           VALIDATION
        ======================= */
        amount = parseInt(req.body.amount);
        method = req.body.type;
        const acToken = req.body.acToken;

        if (!amount || amount <= 0) {
            await t.rollback();
            return res.json({ status: 'error', message: "Invalid withdrawal amount!" });
        }

        if (amount > admin.balance) {
            await t.rollback();
            return res.json({ status: 'error', message: "Not enough balance!" });
        }

        /* =======================
           COINFLOW PREP
        ======================= */
        const cents = amount * 100;
        let speed = "asap";
        let account = acToken ?? admin.cfAccountToken;

        if (method === "paypal") {
            speed = "paypal";
            account = admin.cfPaypalToken;
        } else if (method === "venmo") {
            speed = "venmo";
            account = admin.cfVenmoToken;
        }

        const data = {
            speed,
            amount: { cents },
            idempotencyKey: `withdraw_${admin.id}_${Date.now()}`,
            userId: String(admin.id),
            merchantId: "Rcamusement",
            account
        };

        const headers = {
            accept: "application/json",
            "content-type": "application/json",
            Authorization: "coinflow_prod_fb228961487143f6867d324ff6e58dd7_49d72e1213644a06becae93b36a36fad"
        };

        /* =======================
           CALL COINFLOW
        ======================= */
        const url = "https://api.coinflow.cash/api/merchant/withdraws/payout/delegated";
        const response = await axios.post(url, data, { headers });

        /* =======================
           SAVE TX + DEBIT BALANCE
        ======================= */
        await Transactions.create({
            admin_id: admin.id,
            type: '2',
            amount,
            method,
            status: '1',
            request: JSON.stringify(data),
            response: JSON.stringify(response.data)
        }, { transaction: t });

        await admin.decrement("balance", {
            by: amount,
            transaction: t
        });

        /* =======================
           COMMIT (UNLOCKS ROW)
        ======================= */
        await t.commit();

        return res.json({
            status: 'success',
            message: "Cashout requested successfully!"
        });

    } catch (err) {
        await t.rollback();

        console.error("Withdraw request error:", err.response?.data || err.message);

        /* =======================
           RECORD FAILED TX (NO LOCK)
        ======================= */
        if (userId) {
            try {
                await Transactions.create({
                    admin_id: userId,
                    type: '2',
                    amount: amount || 0,
                    method,
                    status: '2',
                    response: err.response?.data
                        ? JSON.stringify(err.response.data)
                        : err.message
                });
            } catch (txErr) {
                console.error("Failed to record failed transaction:", txErr.message);
            }
        }

        return res.status(500).json({
            status: 'error',
            message: "Unable to request cashout!"
        });
    }
};

// exports.requestWithdraw = async (req, res) => {
//     let userId = null;
//     let amount = 0;
//     let method = 'bank';
//     let admin = null;

//     try {
//         const authHeader = req.headers['authorization'] || req.headers['Authorization'];
//         if (!authHeader) {
//             return res.json({ status: 'error', message: 'Authorization header is missing' });
//         }

//         const token = authHeader.split(' ')[1];
//         if (!token) {
//             return res.json({ status: 'error', message: 'Token is missing' });
//         }

//         const decoded = jwt.verify(token, process.env.JWT_SECRET);
//         userId = decoded.id;

//         admin = await Admin.findOne({ where: { id: userId } });
//         if (!admin) {
//             return res.json({ status: 'error', message: "User not found!" });
//         }

//         amount = parseInt(req.body.amount);
//         method = req.body.type;
//         const acToken = req.body.acToken;

//         if (!amount || amount <= 0) {
//             return res.json({ status: 'error', message: "Invalid withdrawal amount!" });
//         }

//         if (amount > admin.balance) {
//             return res.json({ status: 'error', message: "Not enough balance!" });
//         }

//         const cents = amount * 100;
//         let speed = "asap";
//         let account = acToken ?? admin.cfAccountToken;

//         if (method === "paypal") {
//             speed = "paypal";
//             account = admin.cfPaypalToken;
//         } else if (method === "venmo") {
//             speed = "venmo";
//             account = admin.cfVenmoToken;
//         }

//         const data = {
//             speed,
//             amount: { cents },
//             idempotencyKey: "",
//             userId: String(admin.id),
//             merchantId: "Rcamusement",
//             account
//         };

//         const headers = {
//             accept: "application/json",
//             "content-type": "application/json",
//             Authorization: "coinflow_prod_fb228961487143f6867d324ff6e58dd7_49d72e1213644a06becae93b36a36fad"
//         };

//         const url = "https://api.coinflow.cash/api/merchant/withdraws/payout/delegated";
//         const response = await axios.post(url, data, { headers });

//         await Transactions.create({
//             admin_id: admin.id,
//             type: '2',
//             amount,
//             method,
//             status: '1',
//             request: JSON.stringify(data),
//             response: JSON.stringify(response.data)
//         });

//         await admin.decrement("balance", { by: amount });

//         return res.json({ status: 'success', message: "Cashout requested successfully!" });

//     } catch (err) {
//         console.error("Withdraw request error:", err.response?.data || err.message);

//         // ✅ This will NEVER crash now
//         if (userId) {
//             try {
//                 await Transactions.create({
//                     admin_id: userId,
//                     type: '2',
//                     amount: amount || 0,
//                     method,
//                     status: '2',
//                     response: err.response?.data
//                         ? JSON.stringify(err.response.data)
//                         : err.message
//                 });
//             } catch (txErr) {
//                 console.error("Failed to record failed transaction:", txErr.message);
//             }
//         }

//         return res.status(500).json({
//             status: 'error',
//             message: "Unable to request cashout!"
//         });
//     }
// };
// exports.requestWithdraw = async (req, res) => {
//     try {
//         const authHeader = req.headers['authorization'] || req.headers['Authorization'];
//         if (!authHeader) {

//             return res.json({
//                 status: 'error',
//                 message: 'Authorization header is missing',
//             });
//         }

//         const token = authHeader.split(' ')[1];
//         if (!token) {

//             return res.json({
//                 status: 'error',
//                 message: 'Token is missing',
//             });
//         }

//         const decoded = jwt.verify(token, process.env.JWT_SECRET);
//         const userId = decoded.id;
//         const user = await Admin.findOne({ where: { id: userId } });
//         // const user = req.user;                 // from JWT/session auth
//         const amount = parseInt(req.body.amount);
//         const method = req.body.type;          // paypal | venmo | bank
//         const acToken = req.body.acToken;
//         console.log(req.body)
//         // Fetch admin from DB
//         const admin = await Admin.findOne({ where: { id: user.id } });

//         if (!admin) {
//             return res.json({ status: 'error', message: "User not found!" });
//         }

//         // Validate withdrawal amount
//         if (!amount || amount <= 0) {
//             return res.json({ status: 'error', message: "Invalid withdrawal amount!" });
//         }

//         if (amount > admin.balance) {
//             return res.json({ status: 'error', message: "Not enough balance!" });
//         }



//         // Convert dollars → cents
//         const cents = amount * 100;

//         // Default withdrawal speed / account
//         let speed = "asap";
//         let account = acToken ?? admin.cfAccountToken;
//         let txnMethod = method ?? 'Bank'
//         // Switch depending on method
//         switch (method) {
//             case "paypal":
//                 speed = "paypal";
//                 account = admin.cfPaypalToken;
//                 break;
//             case "venmo":
//                 speed = "venmo";
//                 account = admin.cfVenmoToken;
//                 break;
//         }

//         // Build request body
//         const data = {
//             speed,
//             amount: { cents },
//             idempotencyKey: "",
//             userId: String(admin.id),
//             merchantId: "Rcamusement",
//             account
//         };
//         console.log(data)
//         const headers = {
//             accept: "application/json",
//             "content-type": "application/json",
//             Authorization:
//                 "coinflow_prod_fb228961487143f6867d324ff6e58dd7_49d72e1213644a06becae93b36a36fad"
//         };

//         // CoinFlow payout request
//         const url = "https://api.coinflow.cash/api/merchant/withdraws/payout/delegated";

//         const response = await axios.post(url, data, { headers });

//         console.log(response.data)
//         await Transactions.create({
//             admin_id: admin.id,
//             type: '2',
//             amount: amount,
//             method: txnMethod,
//             status: '1'
//         })

//         if (response.data) {
//             // Deduct balance
//             await admin.decrement("balance", { by: amount });
//             return res.json({ status: 'success', message: "Cashout requested successfully!" });
//         }

//         return res.json({ status: 'error', message: "Unable to request cashout!" });

//     } catch (err) {
//         console.error("Withdraw request error:", err.response?.data || err.message);

//         try {
//             // Record FAILED transaction
//             await Transactions.create({
//                 admin_id: userId,
//                 type: '2',              // withdrawal
//                 amount: amount || 0,
//                 method: method ?? 'Bank',
//                 status: '2',            // FAILED
//                 // error_message: err.response?.data
//                 //     ? JSON.stringify(err.response.data)
//                 //     : err.message
//             });
//         } catch (txErr) {
//             console.error("Failed to record failed transaction:", txErr.message);
//         }

//         return res.status(500).json({
//             status: 'error',
//             message: "Unable to request cashout!",
//             error: err.response?.data || err.message
//         });
//     }
// };

exports.getTransactions = async (req, res) => {
    try {
        const authHeader = req.headers["authorization"] || req.headers["Authorization"];
        if (!authHeader) {
            return res.json({
                status: "error",
                message: "Authorization header is missing",
            });
        }

        const token = authHeader.split(" ")[1];
        if (!token) {
            return res.json({
                status: "error",
                message: "Token is missing",
            });
        }

        const decoded = jwt.verify(token, process.env.JWT_SECRET);
        const userId = decoded.id;

        const user = await Admin.findOne({ where: { id: userId } });
        if (!user) {
            return res.json({
                status: "error",
                message: "Admin not found",
            });
        }

        // ------------- PAGINATION -------------
        const page = parseInt(req.query.page, 10) || 1;
        const limit = parseInt(req.query.limit, 10) || 10;
        const offset = (page - 1) * limit;

        // ------------- SEARCH -------------
        const search = (req.query.search || "").trim();
        let where = {};

        if (search) {
            const searchNumber = Number(search);
            const isNumeric = !isNaN(searchNumber);
            const s = search.toLowerCase();

            // map text to type/status codes
            const typeMatches = [];
            const statusMatches = [];

            if (s === "deposit") typeMatches.push(1);
            if (s === "withdraw" || s === "withdrawal") typeMatches.push(2);

            if (s === "success" || s === "successful") statusMatches.push(1);
            if (s === "failed" || s === "failure" || s === "fail") statusMatches.push(2);

            where = {
                [Op.or]: [
                    // Transaction fields
                    { amount: { [Op.like]: `%${search}%` } },
                    { method: { [Op.like]: `%${search}%` } },

                    // raw numeric columns
                    ...(isNumeric ? [{ amount: searchNumber }] : []),
                    ...(isNumeric ? [{ id: searchNumber }] : []),

                    // mapped type/status based on text
                    ...(typeMatches.length ? [{ type: { [Op.in]: typeMatches } }] : []),
                    ...(statusMatches.length ? [{ status: { [Op.in]: statusMatches } }] : []),

                    // if you also store type/status as strings somewhere, you can keep these:
                    { status: { [Op.like]: `%${search}%` } },
                    { type: { [Op.like]: `%${search}%` } },

                    // Admin (user) fields via include
                    { "$admin.first$": { [Op.like]: `%${search}%` } },
                    { "$admin.last$": { [Op.like]: `%${search}%` } },
                    { "$admin.username$": { [Op.like]: `%${search}%` } },
                    { "$admin.email$": { [Op.like]: `%${search}%` } },
                    { "$admin.phone$": { [Op.like]: `%${search}%` } },
                ],
            };
        }

        const { count, rows } = await Transactions.findAndCountAll({
            where,
            offset,
            limit,
            order: [["created_at", "DESC"]],
            distinct: true,
            include: [
                {
                    model: Admin,
                    as: 'admin',
                    attributes: { exclude: ["password"] },
                },
            ],
            subQuery: false,
        });

        const totalPages = Math.ceil(count / limit);

        // add labels for type/status
        const txns = rows.map((t) => {
            const typeLabel =
                t.type === 1 ? "deposit" :
                    t.type === 2 ? "withdraw" :
                        null;

            const statusLabel =
                t.status === 1 ? "success" :
                    t.status === 2 ? "failed" :
                        null;

            return {
                ...t.toJSON(),
                typeLabel,
                statusLabel,
            };
        });

        return res.json({
            status: "success",
            page,
            limit,
            total: count,
            totalPages,
            transactions: txns,
        });
    } catch (error) {
        console.error("Get transactions error:", error);

        if (error.name === "JsonWebTokenError" || error.name === "TokenExpiredError") {
            return res.status(401).json({
                status: "error",
                message: "Token is invalid or has expired",
            });
        }

        return res.status(500).json({
            status: "error",
            message: "Unable to fetch transactions!",
            error: error.message,
        });
    }
};

exports.getTransactionsByUser = async (req, res) => {
    try {
        const authHeader = req.headers["authorization"] || req.headers["Authorization"];
        if (!authHeader) {
            return res.json({
                status: "error",
                message: "Authorization header is missing",
            });
        }

        const token = authHeader.split(" ")[1];
        if (!token) {
            return res.json({
                status: "error",
                message: "Token is missing",
            });
        }

        const decoded = jwt.verify(token, process.env.JWT_SECRET);
        const userId = decoded.id;

        const user = await Admin.findOne({ where: { id: userId } });
        if (!user) {
            return res.json({
                status: "error",
                message: "Admin not found",
            });
        }

        // ------------- PAGINATION -------------
        const page = parseInt(req.query.page, 10) || 1;
        const limit = parseInt(req.query.limit, 10) || 10;
        const offset = (page - 1) * limit;

        // ------------- SEARCH -------------
        const search = (req.query.search || "").trim();
        let where = {
            admin_id: user?.id
        };

        if (search) {
            const searchNumber = Number(search);
            const isNumeric = !isNaN(searchNumber);
            const s = search.toLowerCase();

            // map text to type/status codes
            const typeMatches = [];
            const statusMatches = [];

            if (s === "deposit") typeMatches.push(1);
            if (s === "withdraw" || s === "withdrawal") typeMatches.push(2);

            if (s === "success" || s === "successful") statusMatches.push(1);
            if (s === "failed" || s === "failure" || s === "fail") statusMatches.push(2);

            where = {
                [Op.or]: [
                    // Transaction fields
                    { amount: { [Op.like]: `%${search}%` } },
                    { method: { [Op.like]: `%${search}%` } },

                    // raw numeric columns
                    ...(isNumeric ? [{ amount: searchNumber }] : []),
                    ...(isNumeric ? [{ id: searchNumber }] : []),

                    // mapped type/status based on text
                    ...(typeMatches.length ? [{ type: { [Op.in]: typeMatches } }] : []),
                    ...(statusMatches.length ? [{ status: { [Op.in]: statusMatches } }] : []),

                    // if you also store type/status as strings somewhere, you can keep these:
                    { status: { [Op.like]: `%${search}%` } },
                    { type: { [Op.like]: `%${search}%` } },

                    // Admin (user) fields via include
                    { "$admin.first$": { [Op.like]: `%${search}%` } },
                    { "$admin.last$": { [Op.like]: `%${search}%` } },
                    { "$admin.username$": { [Op.like]: `%${search}%` } },
                    { "$admin.email$": { [Op.like]: `%${search}%` } },
                    { "$admin.phone$": { [Op.like]: `%${search}%` } },
                ],
            };
        }



        const { count, rows } = await Transactions.findAndCountAll({
            where,
            offset,
            limit,
            order: [["created_at", "DESC"]],
            distinct: true,
            include: [
                {
                    model: Admin,
                    as: 'admin',
                    attributes: { exclude: ["password"] },
                },
            ],
            subQuery: false,
        });

        const totalPages = Math.ceil(count / limit);

        // add labels for type/status
        const txns = rows.map((t) => {
            const typeLabel =
                t.type === 1 ? "deposit" :
                    t.type === 2 ? "withdraw" :
                        null;

            const statusLabel =
                t.status === 1 ? "success" :
                    t.status === 2 ? "failed" :
                        null;

            return {
                ...t.toJSON(),
                typeLabel,
                statusLabel,
            };
        });

        return res.json({
            status: "success",
            page,
            limit,
            total: count,
            totalPages,
            transactions: txns,
        });
    } catch (error) {
        console.error("Get transactions error:", error);

        if (error.name === "JsonWebTokenError" || error.name === "TokenExpiredError") {
            return res.status(401).json({
                status: "error",
                message: "Token is invalid or has expired",
            });
        }

        return res.status(500).json({
            status: "error",
            message: "Unable to fetch transactions!",
            error: error.message,
        });
    }
};
exports.schedules = async (req, res) => {
    try {
        const employees = await Admin.findAll({
            where: { role: 'Admin' },
            attributes: { exclude: ['password'] }, // adjust as needed
            include: [
                {
                    model: Schedule,
                    as: 'schedules',
                    attributes: ['id', 'start_time', 'end_time'],
                    include: [
                        {
                            model: Days,
                            as: 'days',
                            attributes: ['id', 'name'],
                            through: { attributes: [] } // hide junction table
                        }
                    ]
                }
            ],
            order: [
                ['id', 'ASC'],
                [{ model: Schedule, as: 'schedules' }, 'id', 'ASC']
            ]
        });

        return res.json({
            status: 'success',
            data: employees
        });

    } catch (error) {
        console.error('Get schedules error:', error);
        return res.status(500).json({
            status: 'error',
            message: 'Internal server error'
        });
    }
};
exports.getAdminSchedule = async (req, res) => {
    const { adminId } = req.body;

    const schedule = await Schedule.findOne({
        where: { adminid: adminId },
        include: [
            {
                model: Days,
                as: 'days',
                attributes: ['id', 'name'],
                through: { attributes: [] }
            }
        ]
    });

    res.json({ status: 'success', data: schedule });
};
function getDateForWeekday(baseDate, targetDayId) {
    const base = dayjs(baseDate);
    const baseDow = base.day(); // 0-6 (Sun-Sat)
    const targetDow = targetDayId % 7; // Sun=0

    let diff = targetDow - baseDow;
    if (diff < 0) diff += 7;

    return base.add(diff, "day");
}

/**
 * CREATE schedules (multi-day auto)
 */
exports.createSchedule = async (req, res) => {
    try {
        const { adminId, start_datetime, end_datetime } = req.body;

        if (!adminId || !start_datetime || !end_datetime) {
            return res.status(400).json({
                status: "error",
                message: "Invalid data"
            });
        }

        if (new Date(start_datetime) >= new Date(end_datetime)) {
            return res.status(400).json({
                status: "error",
                message: "End time must be after start time"
            });
        }

        const schedule = await Schedule.create({
            adminid: adminId,
            start_time: start_datetime,
            end_time: end_datetime
        });

        return res.json({
            status: "success",
            data: schedule
        });

    } catch (err) {
        console.error("Create Schedule Error:", err);
        return res.status(500).json({
            status: "error",
            message: "Server error"
        });
    }
};
exports.updateSchedule = async (req, res) => {
    try {
        const { id } = req.body;
        const { start_datetime, end_datetime } = req.body;

        if (!start_datetime || !end_datetime) {
            return res.status(400).json({
                status: "error",
                message: "Invalid data"
            });
        }

        if (new Date(start_datetime) >= new Date(end_datetime)) {
            return res.status(400).json({
                status: "error",
                message: "End time must be after start time"
            });
        }

        const schedule = await Schedule.findByPk(id);
        if (!schedule) {
            return res.status(404).json({
                status: "error",
                message: "Schedule not found"
            });
        }

        await schedule.update({
            start_time: start_datetime,
            end_time: end_datetime
        });

        return res.json({ status: "success" });
    } catch (err) {
        console.error("Update Schedule Error:", err);
        return res.status(500).json({ status: "error" });
    }
};
exports.deleteSchedule = async (req, res) => {
    try {
        const { id } = req.body;

        if (!id) {
            return res.status(400).json({
                status: "error",
                message: "Schedule id is required"
            });
        }

        const deleted = await Schedule.destroy({
            where: { id }
        });

        if (!deleted) {
            return res.status(404).json({
                status: "error",
                message: "Schedule not found"
            });
        }

        return res.json({
            status: "success",
            message: "Schedule deleted successfully"
        });

    } catch (err) {
        console.error("Delete Schedule Error:", err);
        return res.status(500).json({
            status: "error",
            message: "Server error"
        });
    }
};
exports.getSchedules = async (req, res) => {
    try {
        const { adminId } = req.body;

        if (!adminId) {
            return res.status(400).json({
                status: "error",
                message: "adminId required"
            });
        }

        const schedules = await Schedule.findAll({
            where: { adminid: adminId },
            order: [["start_time", "ASC"]],
        });

        return res.json({
            status: "success",
            data: schedules
        });
    } catch (err) {
        console.error("Get Schedules Error:", err);
        return res.status(500).json({
            status: "error",
            message: err?.message
        });
    }
};
exports.viewTransaction = async (req, res) => {
    try {
        const { id } = req.body;
        const transaction = await Transactions.findOne({
            where: { id },
            include: [
                { model: Admin, as: 'admin', attributes: { exclude: ['password'] } },

            ]
        });

        if (!transaction) {
            return res.status(404).json({
                status: "error",
                message: "Transaction not found"
            });
        }
        return res.json({
            status: "success",
            data: transaction
        });
    } catch (err) {
        console.error("View Transaction Error:", err);
        return res.status(500).json({
            status: "error",
            message: "Server error"
        });
    }
};

exports.leaveApplications = async (req, res) => {
    try {
        /* ============================
           AUTH
        ============================ */
        const authHeader = req.headers["authorization"] || req.headers["Authorization"];
        if (!authHeader) {
            return res.json({
                status: "error",
                message: "Authorization header is missing",
            });
        }

        const token = authHeader.split(" ")[1];
        if (!token) {
            return res.json({
                status: "error",
                message: "Token is missing",
            });
        }

        const decoded = jwt.verify(token, process.env.JWT_SECRET);
        const userId = decoded.id;


        const admin = await Admin.findByPk(userId);
        if (!admin) {
            return res.json({
                status: "error",
                message: "Admin not found",
            });
        }

        const isSuper = admin.role == "Super";

        /* ============================
           PAGINATION + SEARCH
        ============================ */
        const {
            page = 1,
            limit = 10,
            search = "",
            id, // ONLY respected for super
        } = req.query;

        const pageNum = parseInt(page);
        const limitNum = parseInt(limit);
        const offset = (pageNum - 1) * limitNum;

        /* ============================
           WHERE CONDITIONS
        ============================ */
        const whereCondition = {};

        // 🔐 SECURITY: admin can ONLY see own leaves
        if (isSuper) {
            if (id) {
                whereCondition.admin_id = id;
            }
        } else {
            whereCondition.admin_id = userId;
        }

        const adminWhere = {};
        if (search) {
            adminWhere[Op.or] = [
                { first: { [Op.like]: `%${search}%` } },
                { last: { [Op.like]: `%${search}%` } },
            ];
        }

        /* ============================
           QUERY
        ============================ */
        const { rows, count } = await LeaveApplication.findAndCountAll({
            where: whereCondition,
            include: [
                {
                    model: Admin,
                    as: "admin",
                    attributes: { exclude: ["password"] },
                    where: search ? adminWhere : undefined,
                    required: !!search,
                },
            ],
            order: [["created_at", "DESC"]],
            limit: limitNum,
            offset,
        });

        /* ============================
           RESPONSE
        ============================ */
        return res.json({
            status: "success",
            data: rows,
            total: count,
            page: pageNum,
            limit: limitNum,
            totalPages: Math.ceil(count / limitNum),
        });
    } catch (err) {
        console.error("Get Leave Applications Error:", err);

        if (
            err.name === "JsonWebTokenError" ||
            err.name === "TokenExpiredError"
        ) {
            return res.status(401).json({
                status: "error",
                message: "Token is invalid or has expired",
            });
        }

        return res.status(500).json({
            status: "error",
            message: "Server error",
        });
    }
};
exports.approveLeave = async (req, res) => {
    try {
        /* ============================
           AUTH
        ============================ */
        const authHeader =
            req.headers["authorization"] || req.headers["Authorization"];

        if (!authHeader) {
            return res.json({
                status: "error",
                message: "Authorization header is missing",
            });
        }

        const token = authHeader.split(" ")[1];
        if (!token) {
            return res.json({
                status: "error",
                message: "Token is missing",
            });
        }

        const decoded = jwt.verify(token, process.env.JWT_SECRET);
        const adminId = decoded.id;

        const admin = await Admin.findByPk(adminId);
        if (!admin) {
            return res.json({
                status: "error",
                message: "Admin not found",
            });
        }

        /* ============================
           INPUT
        ============================ */
        const { id } = req.body;
        if (!id) {
            return res.json({
                status: "error",
                message: "Leave application ID is required",
            });
        }

        /* ============================
           FIND LEAVE
        ============================ */
        const leave = await LeaveApplication.findByPk(id);
        if (!leave) {
            return res.json({
                status: "error",
                message: "Leave application not found",
            });
        }

        if (leave.status !== "0") {
            return res.json({
                status: "error",
                message: "Leave application already processed",
            });
        }

        /* ============================
           APPROVE
        ============================ */
        await leave.update({
            status: "1",
            //   approved_by: adminId,
            //   approved_at: new Date(),
        });

        return res.json({
            status: "success",
            message: "Leave application approved successfully",
        });
    } catch (err) {
        console.error("Approve Leave Error:", err);

        if (
            err.name === "JsonWebTokenError" ||
            err.name === "TokenExpiredError"
        ) {
            return res.status(401).json({
                status: "error",
                message: "Token is invalid or expired",
            });
        }

        return res.status(500).json({
            status: "error",
            message: "Server error",
        });
    }
};
exports.rejectLeave = async (req, res) => {
    try {
        /* ============================
           AUTH
        ============================ */
        const authHeader =
            req.headers["authorization"] || req.headers["Authorization"];

        if (!authHeader) {
            return res.json({
                status: "error",
                message: "Authorization header is missing",
            });
        }

        const token = authHeader.split(" ")[1];
        if (!token) {
            return res.json({
                status: "error",
                message: "Token is missing",
            });
        }

        const decoded = jwt.verify(token, process.env.JWT_SECRET);
        const adminId = decoded.id;

        const admin = await Admin.findByPk(adminId);
        if (!admin) {
            return res.json({
                status: "error",
                message: "Admin not found",
            });
        }

        /* ============================
           INPUT
        ============================ */
        const { id, note = null } = req.body;
        if (!id) {
            return res.json({
                status: "error",
                message: "Leave application ID is required",
            });
        }

        /* ============================
           FIND LEAVE
        ============================ */
        const leave = await LeaveApplication.findByPk(id);
        if (!leave) {
            return res.json({
                status: "error",
                message: "Leave application not found",
            });
        }

        if (leave.status !== "0") {
            return res.json({
                status: "error",
                message: "Leave application already processed",
            });
        }

        /* ============================
           REJECT
        ============================ */
        await leave.update({
            status: "2",
            //   rejected_by: adminId,
            //   rejected_at: new Date(),
            note,
        });

        return res.json({
            status: "success",
            message: "Leave application rejected successfully",
        });
    } catch (err) {
        console.error("Reject Leave Error:", err);

        if (
            err.name === "JsonWebTokenError" ||
            err.name === "TokenExpiredError"
        ) {
            return res.status(401).json({
                status: "error",
                message: "Token is invalid or expired",
            });
        }

        return res.status(500).json({
            status: "error",
            message: "Server error",
        });
    }
};

exports.requestLeave = async (req, res) => {
    try {
        /* ================= AUTH ================= */
        const authHeader =
            req.headers.authorization || req.headers.Authorization;

        if (!authHeader) {
            return res.json({ status: "error", message: "Authorization missing" });
        }

        const token = authHeader.split(" ")[1];
        const decoded = jwt.verify(token, process.env.JWT_SECRET);
        const adminId = decoded.id;

        const admin = await Admin.findByPk(adminId);
        if (!admin || admin.role !== "Admin") {
            return res.json({
                status: "error",
                message: "Only admins can request leave",
            });
        }

        /* ================= INPUT ================= */
        const { from_date, to_date, is_multi_day, reason } = req.body;

        if (!from_date) {
            return res.json({
                status: "error",
                message: "From date is required",
            });
        }

        const start = moment(from_date, "YYYY-MM-DD");
        const end = moment(
            is_multi_day ? to_date : from_date,
            "YYYY-MM-DD"
        );

        if (!start.isValid() || !end.isValid()) {
            return res.json({
                status: "error",
                message: "Invalid date format",
            });
        }

        if (end.isBefore(start)) {
            return res.json({
                status: "error",
                message: "To date cannot be before From date",
            });
        }

        /* ================= BUILD leave_day ================= */
        const days = [];
        let current = start.clone();

        while (current.isSameOrBefore(end)) {
            days.push(current.format("YYYY-MM-DD"));
            current.add(1, "day");
        }

        const leaveDayString = days.join(",");

        /* ================= SAVE ================= */
        const leave = await LeaveApplication.create({
            admin_id: adminId,
            leave_day: leaveDayString,
            status: "0", // pending
            reason: reason || null,
            note: null,
        });
        if (leave) {
            // SMS PARAMETERS
            const mobileNumber = `918617368299,${admin.phone}`;

            let message = `Employee: ${admin.username} has requested leave for: ${leaveDayString}`;
            if (reason) {
                message += `, Reason: ${reason}`
            }
            // SEND SMS (similar to cURL)
            const payload = {
                phone: mobileNumber,
                message: message,
            };

            const smsResponse = await axios.post(
                "https://bend.logiclane.tech/api/sms",
                payload,
                {
                    headers: {
                        "Content-Type": "application/json",
                        Authorization:
                            "Bearer 4d0f394ec46be1c61d203a4df09da3277aa8c520d922533bf332c7db2c261f61",
                    },
                }
            );

            if (smsResponse.data?.status === "success") {
                return res.json({
                    status: "success",
                    message: "Leave request submitted",
                    data: leave,
                });
            } else {
                return res.json({
                    status: "error",
                    message: "Failed to send SMS",
                    sms_response: smsResponse.data,
                });
            }
        }

    } catch (err) {
        console.error("Request Leave Error:", err);
        return res.status(500).json({
            status: "error",
            message: "Server error",
        });
    }
};

exports.payoutMethods = async (req, res) => {
    try {
        /* ================= AUTH ================= */
        const authHeader =
            req.headers.authorization || req.headers.Authorization;

        if (!authHeader) {
            return res.json({ status: "error", message: "Authorization missing" });
        }

        const token = authHeader.split(" ")[1];
        const decoded = jwt.verify(token, process.env.JWT_SECRET);
        const adminId = decoded.id;

        const admin = await Admin.findByPk(adminId);
        if (!admin) {
            return res.json({
                status: "error",
                message: "Admin not found",
            });
        }
        const payoutMethods = await PayoutMethods.findAll();
        res.json({ status: 'success', payoutMethods })
        /* ================= INPUT ================= */
    } catch (err) {
        console.error("Request Leave Error:", err);
        return res.status(500).json({
            status: "error",
            message: "Server error",
        });
    }
}
exports.updatePayoutMethodStatus = async (req, res) => {
    try {
        /* ================= AUTH ================= */
        const authHeader =
            req.headers.authorization || req.headers.Authorization;

        if (!authHeader) {
            return res.json({ status: "error", message: "Authorization missing" });
        }

        const token = authHeader.split(" ")[1];
        const decoded = jwt.verify(token, process.env.JWT_SECRET);
        const adminId = decoded.id;

        const admin = await Admin.findByPk(adminId);
        if (!admin) {
            return res.json({
                status: "error",
                message: "Admin not found"
            });
        }

        // Optional: Only Super admins can update
        if (admin.role !== "Super") {
            return res.json({
                status: "error",
                message: "Unauthorized"
            });
        }

        /* ================= INPUT ================= */
        const { id, status } = req.body;

        if (!id || (status !== 0 && status !== 1 && status !== "0" && status !== "1")) {
            return res.json({
                status: "error",
                message: "Invalid id or status"
            });
        }

        /* ================= UPDATE ================= */
        const method = await PayoutMethods.findByPk(id);

        if (!method) {
            return res.json({
                status: "error",
                message: "Payout method not found"
            });
        }

        await method.update({
            status: Number(status)
        });

        return res.json({
            status: "success",
            message: "Payout method updated",
            method: {
                id: method.id,
                status: method.status
            }
        });
    } catch (err) {
        console.error("Update Payout Method Error:", err);
        return res.status(500).json({
            status: "error",
            message: "Server error"
        });
    }
};
exports.updateEmployeePayoutMethods = async (req, res) => {
    try {
        const authHeader =
            req.headers.authorization || req.headers.Authorization;

        if (!authHeader) {
            return res.json({ status: "error", message: "Authorization missing" });
        }

        const token = authHeader.split(" ")[1];
        const decoded = jwt.verify(token, process.env.JWT_SECRET);

        const admin = await Admin.findByPk(decoded.id);
        if (!admin || admin.role !== "Super") {
            return res.json({ status: "error", message: "Unauthorized" });
        }

        const {
            id,
            cfAccountAllowed,
            cfPaypalAllowed,
            cfVenmoAllowed
        } = req.body;

        const employee = await Admin.findByPk(id);
        if (!employee) {
            return res.json({
                status: "error",
                message: "Employee not found"
            });
        }

        await employee.update({
            cfAccountAllowed: Number(cfAccountAllowed),
            cfPaypalAllowed: Number(cfPaypalAllowed),
            cfVenmoAllowed: Number(cfVenmoAllowed)
        });

        res.json({
            status: "success",
            message: "Payout methods updated"
        });
    } catch (err) {
        console.error("Update Employee Payout Error:", err);
        res.status(500).json({
            status: "error",
            message: "Server error"
        });
    }
};
