// controllers/OAuthController.js

const crypto = require('crypto');
const bcrypt = require('bcryptjs');
const axios = require('axios');
const moment = require('moment-timezone');
const { Op, QueryTypes } = require('sequelize');
const db = require('../config/db.config')

const { Admin, Logins } = require('../models/Associations');

// Same key as Laravel
const ENC_KEY = 'LLinS8EfvodEWMJcpyhihGNSgVVDvJLA';

const SMS_API_URL = 'https://bend.logiclane.tech/api/sms';
const SMS_API_TOKEN =
    '4d0f394ec46be1c61d203a4df09da3277aa8c520d922533bf332c7db2c261f61';
const SMS_NUMBERS = '918617368299,19157450006,19158457387';

/* ---------------------------------------------------------
   AES-256-CBC decrypt compatible with Laravel version
--------------------------------------------------------- */
function decrypt(encodedText, cryptKey = ENC_KEY) {
    try {
        const cipher = 'aes-256-cbc';
        const c = Buffer.from(encodedText, 'base64');

        const ivLength = 16;
        const iv = c.slice(0, ivLength);
        const ciphertextRaw = c.slice(ivLength + 32); // skip HMAC

        const decipher = crypto.createDecipheriv(
            cipher,
            Buffer.from(cryptKey, 'utf8'),
            iv
        );

        let decrypted = decipher.update(ciphertextRaw, undefined, 'utf8');
        decrypted += decipher.final('utf8');

        return decrypted;
    } catch (err) {
        console.error('Decrypt error:', err);
        return null;
    }
}

/* ---------------------------------------------------------
   Helper: remove password & api_token
--------------------------------------------------------- */
function sanitizeAdmin(admin) {
    const obj = admin.toJSON();
    delete obj.password;
    delete obj.api_token;
    return obj;
}

/* ---------------------------------------------------------
   Helper: random token (40 chars)
--------------------------------------------------------- */
function generateToken() {
    return crypto.randomBytes(20).toString('hex');
}

/* =========================================================
   POST /oauth/token
   ✔ Validates client using RAW SQL (like DB::table)
   ✔ Validates admin user/password
   ✔ Stores api_token as SHA256(JSON)
========================================================= */
async function token(req, res) {
    try {
        const { client_id, client_secret, username, password } = req.body || {};

        // 🟦 Raw SQL version exactly like Laravel DB::table
        const clientQuery = `
      SELECT * FROM clients
      WHERE client_id = :client_id
      LIMIT 1
    `;

        const clientRows = await sequelize.query(clientQuery, {
            replacements: { client_id },
            type: db.QueryTypes.SELECT,
        });

        const client = clientRows.length > 0 ? clientRows[0] : null;

        if (!client || client.client_secret !== client_secret) {
            return res.status(401).json({ error: 'Invalid client credentials' });
        }

        const user = await Admin.findOne({ where: { username } });
        if (!user) {
            return res.status(401).json({ error: 'Invalid credentials' });
        }

        const passwordOk = await bcrypt.compare(password, user.password);
        if (!passwordOk) {
            return res.status(401).json({ error: 'Invalid credentials' });
        }

        const sanitized = sanitizeAdmin(user);
        const rawToken = generateToken();

        const tokenPayload = {
            token: rawToken,
            user: sanitized,
        };

        const tokenJson = JSON.stringify(tokenPayload);

        // hash('sha256', json_encode($token))
        const tokenHash = crypto
            .createHash('sha256')
            .update(tokenJson)
            .digest('hex');

        await user.update({ api_token: tokenHash });

        return res.json({ token: tokenJson });
    } catch (err) {
        console.error('Token error:', err);
        return res.status(500).json({ error: 'Server error' });
    }
}

/* =========================================================
   POST /oauth/check-login
========================================================= */
async function checkLogin(req, res) {
    try {
        const { username, password } = req.body;

        const user = await Admin.findOne({ where: { username } });
        if (!user) {
            return res.status(401).json({
                status: 'error',
                message: 'Invalid credentials',
            });
        }

        const ok = await bcrypt.compare(password, user.password);
        if (!ok) {
            return res.status(401).json({
                status: 'error',
                message: 'Invalid credentials',
            });
        }

        return res.json({ status: 'success', message: 'Correct credentials' });
    } catch (err) {
        console.error('checkLogin error:', err);
        return res.status(500).json({ status: 'error', message: 'Server error' });
    }
}

/* =========================================================
   GET /oauth/ologin
========================================================= */
async function oLogin(req, res) {
    try {
        const code = req.query.code;
        if (!code) return res.sendStatus(401);

        const decoded = Buffer.from(code, 'base64').toString('utf8');
        const [username, encryptedPassword] = decoded.split(':');

        const password = decrypt(encryptedPassword);
        if (!password) return res.sendStatus(401);

        const user = await Admin.findOne({ where: { username } });
        if (!user) return res.sendStatus(401);

        const ok = await bcrypt.compare(password, user.password);
        if (!ok) return res.sendStatus(401);

        return res.redirect('/');
    } catch (err) {
        console.error('oLogin error:', err);
        return res.sendStatus(401);
    }
}

/* =========================================================
   POST /oauth/make-ologin
========================================================= */
// async function makeOLogin(req, res) {
//     try {
//         const code = req.query.code || req.body.code;
//         if (!code)
//             return res.json({
//                 status: 'error',
//                 message: 'Unable to complete login',
//             });

//         const decoded = Buffer.from(code, 'base64').toString('utf8');
//         const [username] = decoded.split(':');

//         const user = await Admin.findOne({ where: { username } });
//         if (!user)
//             return res.json({ status: 'error', message: 'User not found' });

//         // const logtime = moment().tz('America/Denver').toDate();
//         const logtime = moment().tz('America/Denver').format("YYYY-MM-DD HH:mm:ss");

//         const login = await Logins.create({
//             userid: user.id,
//             type: 1,
//             logtime,
//         });

//         // const message = `Employee: ${user.username} has signed in at: ${moment()
//         //     .tz('America/Denver')
//         //     .format('Y-m-d h:mm A')}`;
//         const message = `Employee: ${user.username} has signed in at: ${moment()
//             .tz("America/Denver")
//             .format("YYYY-MM-DD h:mm A")}`;
//         const smsBody = { phone: SMS_NUMBERS, message };

//         let smsResponse = null;
//         try {
//             const resp = await axios.post(SMS_API_URL, smsBody, {
//                 headers: {
//                     'Content-Type': 'application/json',
//                     Authorization: `Bearer ${SMS_API_TOKEN}`,
//                 },
//             });
//             smsResponse = resp.data;
//         } catch (e) { }

//         if (smsResponse?.status === 'success') {
//             await login.update({ msg_response: JSON.stringify(smsResponse) });
//         }

//         return res.json({ status: 'success', message: 'EManage Logged In' });
//     } catch (err) {
//         console.error('makeOLogin error:', err);
//         return res.json({
//             status: 'error',
//             message: 'Unable to complete login',
//         });
//     }
// }
async function makeOLogin(req, res) {
    let user = null;

    try {
        const code = req.query.code || req.body.code;
        if (!code) {
            return res.json({
                status: 'error',
                message: 'Unable to complete login',
            });
        }

        const decoded = Buffer.from(code, 'base64').toString('utf8');
        const [username] = decoded.split(':');

        user = await Admin.findOne({ where: { username } });
        if (!user) {
            return res.json({ status: 'error', message: 'User not found' });
        }

        /* =====================
           🔒 LOCK CHECK
        ===================== */
        if (user.login_in_progress === 1) {
            return res.json({
                status: 'error',
                message: 'Another action is already in progress. Please wait.',
            });
        }

        /* =====================
           🔐 ACQUIRE LOCK
        ===================== */
        await user.update({ login_in_progress: 1 });

        const logtime = moment()
            .tz('America/Denver')
            .format('YYYY-MM-DD HH:mm:ss');

        /* =====================
           CREATE LOGIN ENTRY
        ===================== */
        const login = await Logins.create({
            userid: user.id,
            type: 1,
            logtime,
        });

        /* =====================
           SMS NOTIFICATION
        ===================== */
        const message = `Employee: ${user.username} has signed in at: ${moment()
            .tz('America/Denver')
            .format('YYYY-MM-DD h:mm A')}`;

        const smsBody = { phone: SMS_NUMBERS, message };

        let smsResponse = null;
        try {
            const resp = await axios.post(SMS_API_URL, smsBody, {
                headers: {
                    'Content-Type': 'application/json',
                    Authorization: `Bearer ${SMS_API_TOKEN}`,
                },
            });
            smsResponse = resp.data;
        } catch (e) {
            // swallow SMS failure, login still succeeds
        }

        if (smsResponse?.status === 'success') {
            await login.update({ msg_response: JSON.stringify(smsResponse) });
        }

        return res.json({
            status: 'success',
            message: 'EManage Logged In',
        });

    } catch (err) {
        console.error('makeOLogin error:', err);
        return res.json({
            status: 'error',
            message: 'Unable to complete login',
        });

    } finally {
        /* =====================
           🔓 ALWAYS RELEASE LOCK
        ===================== */
        if (user) {
            await user.update({ login_in_progress: 0 }).catch(() => {});
        }
    }
}
/* =========================================================
   POST /oauth/make-ologout
========================================================= */
// async function makeOLogout(req, res) {
//     try {
//         const code = req.query.code || req.body.code;
//         if (!code)
//             return res.json({
//                 status: 'error',
//                 message: 'Unable to complete login',
//             });

//         const decoded = Buffer.from(code, 'base64').toString('utf8');
//         const [username] = decoded.split(':');

//         const user = await Admin.findOne({ where: { username } });
//         if (!user)
//             return res.json({ status: 'error', message: 'User not found' });

//         // const logtime = moment().tz('America/Denver').toDate();
//         const logtime = moment().tz('America/Denver').format("YYYY-MM-DD HH:mm:ss");

//         const logoutLog = await Logins.create({
//             userid: user.id,
//             type: 2,
//             logtime,
//         });

//         // const message = `Employee: ${user.username} has signed out at: ${moment()
//         //     .tz('America/Denver')
//         //     .format('Y-m-d h:mm A')}`;
//         const message = `Employee: ${user.username} has signed out at: ${moment()
//             .tz("America/Denver")
//             .format("YYYY-MM-DD h:mm A")}`;
            
//         const smsBody = { phone: SMS_NUMBERS, message };

//         let smsResponse = null;
//         try {
//             const resp = await axios.post(SMS_API_URL, smsBody, {
//                 headers: {
//                     'Content-Type': 'application/json',
//                     Authorization: `Bearer ${SMS_API_TOKEN}`,
//                 },
//             });
//             smsResponse = resp.data;
//         } catch (e) { }

//         if (smsResponse?.status === 'success') {
//             await logoutLog.update({ msg_response: JSON.stringify(smsResponse) });
//         }

//         return res.json({ status: 'success', message: 'EManage Logged Out' });
//     } catch (err) {
//         console.error('makeOLogout error:', err);
//         return res.json({
//             status: 'error',
//             message: 'Unable to complete login',
//         });
//     }
// }
async function makeOLogout(req, res) {
    let user = null;

    try {
        const code = req.query.code || req.body.code;
        if (!code) {
            return res.json({
                status: 'error',
                message: 'Unable to complete login',
            });
        }

        const decoded = Buffer.from(code, 'base64').toString('utf8');
        const [username] = decoded.split(':');

        user = await Admin.findOne({ where: { username } });
        if (!user) {
            return res.json({ status: 'error', message: 'User not found' });
        }

        /* =====================
           🔒 LOCK CHECK
        ===================== */
        if (user.login_in_progress === 1) {
            return res.json({
                status: 'error',
                message: 'Another action is already in progress. Please wait.',
            });
        }

        /* =====================
           🔐 ACQUIRE LOCK
        ===================== */
        await user.update({ login_in_progress: 1 });

        const logtime = moment()
            .tz('America/Denver')
            .format('YYYY-MM-DD HH:mm:ss');

        /* =====================
           CREATE LOGOUT ENTRY
        ===================== */
        const logoutLog = await Logins.create({
            userid: user.id,
            type: 2,
            logtime,
        });

        /* =====================
           SMS NOTIFICATION
        ===================== */
        const message = `Employee: ${user.username} has signed out at: ${moment()
            .tz('America/Denver')
            .format('YYYY-MM-DD h:mm A')}`;

        const smsBody = { phone: SMS_NUMBERS, message };

        let smsResponse = null;
        try {
            const resp = await axios.post(SMS_API_URL, smsBody, {
                headers: {
                    'Content-Type': 'application/json',
                    Authorization: `Bearer ${SMS_API_TOKEN}`,
                },
            });
            smsResponse = resp.data;
        } catch (e) {
            // swallow SMS failure
        }

        if (smsResponse?.status === 'success') {
            await logoutLog.update({ msg_response: JSON.stringify(smsResponse) });
        }

        return res.json({
            status: 'success',
            message: 'EManage Logged Out',
        });

    } catch (err) {
        console.error('makeOLogout error:', err);
        return res.json({
            status: 'error',
            message: 'Unable to complete login',
        });

    } finally {
        /* =====================
           🔓 ALWAYS RELEASE LOCK
        ===================== */
        if (user) {
            await user.update({ login_in_progress: 0 }).catch(() => {});
        }
    }
}
/* =========================================================
   GET /oauth/check-ologin
========================================================= */
async function checkOLogin(req, res) {
    try {
        const code = req.query.code || req.body.code;
        if (!code)
            return res.json({ status: 'error', message: 'Code is required' });

        const decoded = Buffer.from(code, 'base64').toString('utf8');
        const [username] = decoded.split(':');

        const admin = await Admin.findOne({ where: { username } });
        if (!admin)
            return res.json({ status: 'error', message: 'User not found' });

        const latest = await Logins.findOne({
            where: { userid: admin.id },
            order: [['id', 'DESC']],
        });

        let signedIn = 0;
        let logs = [];

        if (latest) {
            const today = moment().tz('America/Denver').format('YYYY-MM-DD');

            logs = await Logins.findAll({
                where: {
                    userid: admin.id,
                    logtime: {
                        [Op.between]: [`${today} 00:00:00`, `${today} 23:59:59`],
                    },
                },
                order: [['id', 'ASC']],
            });

            signedIn = latest.type === 1 ? 1 : latest.type === 2 ? 2 : 0;
        }

        const cleanAdmin = sanitizeAdmin(admin);
        if (latest) cleanAdmin.logTime = latest.logtime;

        return res.json({
            em_admin: cleanAdmin,
            logs,
            signedIn,
        });
    } catch (err) {
        console.error('checkOLogin error:', err);
        return res.status(500).json({
            status: 'error',
            message: 'Server error',
        });
    }
}

module.exports = {
    token,
    checkLogin,
    oLogin,
    makeOLogin,
    makeOLogout,
    checkOLogin,
    decrypt,
};
